#!/bin/bash

###############################################################################
# MCC Monitoring Stack - Automated Setup Script
# This script will automatically set up the complete monitoring stack
# Usage: sudo ./setup.sh [--clean]
#        --clean : Remove all data and start fresh
###############################################################################

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Parse arguments
CLEAN_INSTALL=false
for arg in "$@"; do
    case $arg in
        --clean)
            CLEAN_INSTALL=true
            shift
            ;;
    esac
done

# Auto-detect network interface (primary interface with default route)
detect_network_interface() {
    # Try to get interface with default route
    local iface=$(ip route | grep default | awk '{print $5}' | head -1)
    
    # If not found, try getting first non-lo interface
    if [ -z "$iface" ]; then
        iface=$(ip link show | grep -E '^[0-9]+:' | grep -v 'lo:' | head -1 | awk -F: '{print $2}' | tr -d ' ')
    fi
    
    echo "$iface"
}

# Auto-detect MCC container name
detect_mcc_container() {
    # Look for container with 'mcc' in name (case insensitive) that's not part of monitoring stack
    local container=$(docker ps --format '{{.Names}}' 2>/dev/null | grep -i 'mcc' | grep -v 'mcc-' | head -1)
    
    # If not found, look for IoT Edge connected cache container
    if [ -z "$container" ]; then
        container=$(docker ps --format '{{.Names}}' 2>/dev/null | grep -iE 'cache|connected' | head -1)
    fi
    
    echo "$container"
}

# Auto-detect MCC log path
detect_mcc_log_path() {
    local container="$1"
    local log_path=""
    
    # Try common MCC log paths
    for path in "/media/data/node1/logs" "/var/mcc/logs" "/media/data/logs" "/mnt/data/logs"; do
        if [ -d "$path" ]; then
            log_path="$path"
            break
        fi
    done
    
    # If container exists, try to find nginx log path from container
    if [ -n "$container" ] && docker ps --format '{{.Names}}' | grep -q "^${container}$"; then
        local container_log=$(docker exec "$container" find /media -name "*.log" -path "*/logs/*" 2>/dev/null | head -1 | xargs dirname 2>/dev/null)
        if [ -n "$container_log" ]; then
            log_path="$container_log"
        fi
    fi
    
    echo "${log_path:-/media/data/node1/logs}"
}

# Auto-detect cache disk path
detect_cache_disk_path() {
    local disk_path=""
    
    # Check common cache disk mount points
    for path in "/media/data" "/mnt/data" "/mnt/cache" "/var/mcc"; do
        if [ -d "$path" ]; then
            disk_path="$path"
            break
        fi
    done
    
    echo "${disk_path:-/media/data}"
}

# Configuration Variables - Auto-detect or use environment variables
DETECTED_INTERFACE=$(detect_network_interface)
DETECTED_MCC_CONTAINER=$(detect_mcc_container)

MCC_CONTAINER_NAME="${MCC_CONTAINER_NAME:-${DETECTED_MCC_CONTAINER:-MCC}}"
MCC_LOG_PATH="${MCC_LOG_PATH:-$(detect_mcc_log_path "$MCC_CONTAINER_NAME")}"
NETWORK_INTERFACE="${NETWORK_INTERFACE:-${DETECTED_INTERFACE:-eth0}}"
CACHE_DISK_PATH="${CACHE_DISK_PATH:-$(detect_cache_disk_path)}"
GRAFANA_ADMIN_PASSWORD="${GRAFANA_ADMIN_PASSWORD:-admin}"
INFLUXDB_TOKEN="${INFLUXDB_TOKEN:-mcc-monitoring-token-2025}"

echo -e "${GREEN}=========================================${NC}"
echo -e "${GREEN}MCC Monitoring Stack Setup${NC}"
echo -e "${GREEN}=========================================${NC}"
echo ""

if [ "$CLEAN_INSTALL" = true ]; then
    echo -e "${RED}⚠ CLEAN INSTALL: All existing data will be removed!${NC}"
    echo ""
fi

# Show auto-detected configuration
echo -e "${BLUE}Auto-Detected Configuration:${NC}"
echo -e "  MCC Container Name: $MCC_CONTAINER_NAME ${DETECTED_MCC_CONTAINER:+(auto-detected)}"
echo -e "  MCC Log Path: $MCC_LOG_PATH"
echo -e "  Network Interface: $NETWORK_INTERFACE ${DETECTED_INTERFACE:+(auto-detected)}"
echo -e "  Cache Disk Path: $CACHE_DISK_PATH"
echo ""
echo -e "${YELLOW}To override, set environment variables before running:${NC}"
echo -e "  NETWORK_INTERFACE=eth0 MCC_CONTAINER_NAME=MCC ./setup.sh"
echo ""

# Check if running as root
if [ "$EUID" -ne 0 ]; then 
    echo -e "${RED}Please run as root (sudo ./setup.sh)${NC}"
    exit 1
fi

# Get script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR"

# Check for docker and docker-compose
echo -e "${YELLOW}[1/10] Checking prerequisites...${NC}"
if ! command -v docker &> /dev/null; then
    echo -e "${RED}Docker is not installed. Please install Docker first.${NC}"
    exit 1
fi

# Check for docker-compose or docker compose
if command -v docker-compose &> /dev/null; then
    DOCKER_COMPOSE="docker-compose"
elif docker compose version &> /dev/null 2>&1; then
    DOCKER_COMPOSE="docker compose"
else
    echo -e "${YELLOW}Docker Compose not found. Installing Docker Compose plugin...${NC}"
    
    # Install Docker Compose plugin
    apt-get update -qq
    apt-get install -y -qq docker-compose-plugin 2>/dev/null || {
        # If apt package not available, install manually
        echo -e "${YELLOW}Installing Docker Compose manually...${NC}"
        COMPOSE_VERSION=$(curl -s https://api.github.com/repos/docker/compose/releases/latest | grep -oP '"tag_name": "\K(.*)(?=")')
        if [ -z "$COMPOSE_VERSION" ]; then
            COMPOSE_VERSION="v2.24.0"  # Fallback version
        fi
        
        mkdir -p /usr/local/lib/docker/cli-plugins
        curl -SL "https://github.com/docker/compose/releases/download/${COMPOSE_VERSION}/docker-compose-linux-$(uname -m)" -o /usr/local/lib/docker/cli-plugins/docker-compose
        chmod +x /usr/local/lib/docker/cli-plugins/docker-compose
        
        # Also create symlink for docker-compose command
        ln -sf /usr/local/lib/docker/cli-plugins/docker-compose /usr/local/bin/docker-compose 2>/dev/null || true
    }
    
    # Verify installation
    if docker compose version &> /dev/null 2>&1; then
        DOCKER_COMPOSE="docker compose"
        echo -e "${GREEN}✓ Docker Compose installed successfully${NC}"
    elif command -v docker-compose &> /dev/null; then
        DOCKER_COMPOSE="docker-compose"
        echo -e "${GREEN}✓ Docker Compose installed successfully${NC}"
    else
        echo -e "${RED}Failed to install Docker Compose. Please install manually.${NC}"
        exit 1
    fi
fi

echo -e "${GREEN}✓ Docker and Docker Compose found (using: $DOCKER_COMPOSE)${NC}"

# Check if MCC container exists
echo -e "${YELLOW}[2/10] Checking MCC container...${NC}"
if docker ps -a --format '{{.Names}}' | grep -q "^${MCC_CONTAINER_NAME}$"; then
    echo -e "${GREEN}✓ MCC container '$MCC_CONTAINER_NAME' found${NC}"
else
    echo -e "${YELLOW}⚠ MCC container '$MCC_CONTAINER_NAME' not found. BGP monitoring may not work.${NC}"
fi

# Check if MCC log path exists
if [ -d "$MCC_LOG_PATH" ]; then
    echo -e "${GREEN}✓ MCC log path exists: $MCC_LOG_PATH${NC}"
else
    echo -e "${YELLOW}⚠ MCC log path not found: $MCC_LOG_PATH${NC}"
    echo -e "${YELLOW}  Creating directory...${NC}"
    mkdir -p "$MCC_LOG_PATH"
fi

# Generate dynamic UIDs for datasources
echo -e "${YELLOW}[3/10] Generating dynamic datasource UIDs...${NC}"
INFLUXDB_UID=$(cat /dev/urandom | tr -dc 'A-Z0-9' | fold -w 17 | head -n 1)
LOKI_UID=$(cat /dev/urandom | tr -dc 'A-Z0-9' | fold -w 17 | head -n 1)

echo "Generated InfluxDB UID: $INFLUXDB_UID"
echo "Generated Loki UID: $LOKI_UID"

# Update datasource configurations
echo -e "${YELLOW}[4/10] Updating datasource configurations...${NC}"

# Update InfluxDB datasource
cat > grafana/provisioning/datasources/influxdb.yml << EOF
apiVersion: 1

datasources:
  - name: InfluxDB
    type: influxdb
    uid: $INFLUXDB_UID
    access: proxy
    url: http://influxdb:8086
    jsonData:
      version: Flux
      organization: mcc
      defaultBucket: metrics
      tlsSkipVerify: true
    secureJsonData:
      token: $INFLUXDB_TOKEN
    isDefault: false
    editable: false
EOF

# Update Loki datasource
cat > grafana/provisioning/datasources/loki.yml << EOF
apiVersion: 1

datasources:
  - name: Loki
    type: loki
    uid: $LOKI_UID
    access: proxy
    url: http://loki:3100
    jsonData:
      maxLines: 1000
    isDefault: true
    editable: false
EOF

# Remove any old datasources.yml to avoid conflicts
rm -f grafana/provisioning/datasources/datasources.yml 2>/dev/null || true

echo -e "${GREEN}✓ Datasource configurations updated${NC}"

# Update dashboard with dynamic UIDs
echo -e "${YELLOW}[5/10] Updating dashboard with dynamic UIDs...${NC}"

# Backup original dashboard if backup doesn't exist
if [ ! -f grafana/dashboards/mcc_dashboard.json.original ]; then
    cp grafana/dashboards/mcc_dashboard.json grafana/dashboards/mcc_dashboard.json.original
fi

# Restore from original to ensure clean replacement
cp grafana/dashboards/mcc_dashboard.json.original grafana/dashboards/mcc_dashboard.json

# Replace ALL UIDs in dashboard using global replacement
sed -i "s/P951FEA4DE68E13C5/$INFLUXDB_UID/g" grafana/dashboards/mcc_dashboard.json
sed -i "s/P8E80F9AEF21F6940/$LOKI_UID/g" grafana/dashboards/mcc_dashboard.json

# Also replace any other hardcoded UIDs that might exist
sed -i "s/XWG9472KCVHAA0C2C/$INFLUXDB_UID/g" grafana/dashboards/mcc_dashboard.json

# Update network interface in dashboard queries
sed -i "s/ens18/$NETWORK_INTERFACE/g" grafana/dashboards/mcc_dashboard.json

echo -e "${GREEN}✓ Dashboard updated with dynamic UIDs and network interface: $NETWORK_INTERFACE${NC}"

# Update docker-compose.yml with MCC log path
echo -e "${YELLOW}[6/10] Updating docker-compose configuration...${NC}"

# Update promtail volume path
sed -i "s|/media/data/node1/logs:/var/log/mcc:ro|$MCC_LOG_PATH:/var/log/mcc:ro|g" docker-compose.yml

# Update telegraf network interface
sed -i "s/interfaces = \[\"ens18\"\]/interfaces = [\"$NETWORK_INTERFACE\"]/g" telegraf/telegraf.conf

echo -e "${GREEN}✓ Docker compose and Telegraf configuration updated${NC}"

# Update BGP scripts with MCC container name
echo -e "${YELLOW}[7/10] Updating BGP monitoring scripts...${NC}"

# Update bgp_status.sh
cat > telegraf/scripts/bgp_status.sh << EOFBGP
#!/bin/bash

# BGP Status monitoring script for MCC container
MCC_CONTAINER_NAME="$MCC_CONTAINER_NAME"

# Get BGP protocol name (case-insensitive)
BGP_PROTO=\$(docker exec \$MCC_CONTAINER_NAME birdc show protocols 2>/dev/null | grep -i bgp | awk '{print \$1}' | head -1)

if [ -z "\$BGP_PROTO" ]; then
    echo "bgp_status,host=mcc-node state=\"down\",state_numeric=0i"
    echo "bgp_routes,host=mcc-node learned=0i"
    exit 0
fi

# Get BGP state from protocols list
BGP_INFO=\$(docker exec \$MCC_CONTAINER_NAME birdc show protocols 2>/dev/null | grep -i "\$BGP_PROTO" | awk '{print \$6}')

# Get route counts
ROUTE_OUTPUT=\$(docker exec \$MCC_CONTAINER_NAME birdc show route protocol \$BGP_PROTO count 2>/dev/null)
ROUTES_LEARNED=\$(echo "\$ROUTE_OUTPUT" | grep -oP '^\d+' | head -1)
TOTAL_ROUTES=\$(echo "\$ROUTE_OUTPUT" | grep -oP 'for \K\d+' | head -1)

# Get detailed neighbor info
NEIGHBOR_IP=\$(docker exec \$MCC_CONTAINER_NAME birdc show protocols all \$BGP_PROTO 2>/dev/null | grep "Neighbor address" | awk '{print \$3}')
NEIGHBOR_AS=\$(docker exec \$MCC_CONTAINER_NAME birdc show protocols all \$BGP_PROTO 2>/dev/null | grep "Neighbor AS" | awk '{print \$3}')

# Default values
BGP_STATE=\${BGP_INFO:-"down"}
ROUTES_LEARNED=\${ROUTES_LEARNED:-0}
TOTAL_ROUTES=\${TOTAL_ROUTES:-0}
NEIGHBOR_AS=\${NEIGHBOR_AS:-0}

# Convert BGP status to numeric
if [ "\$BGP_STATE" = "Established" ]; then
    STATUS_NUM=1
else
    STATUS_NUM=0
fi

# Output in InfluxDB line protocol
echo "bgp_status,host=mcc-node,protocol=\$BGP_PROTO state=\"\$BGP_STATE\",state_numeric=\${STATUS_NUM}i"
echo "bgp_routes,host=mcc-node,protocol=\$BGP_PROTO learned=\${ROUTES_LEARNED}i,total=\${TOTAL_ROUTES}i"

if [ -n "\$NEIGHBOR_IP" ]; then
    echo "bgp_neighbor,host=mcc-node,neighbor=\$NEIGHBOR_IP,protocol=\$BGP_PROTO as_number=\${NEIGHBOR_AS}i,prefixes=\${ROUTES_LEARNED}i"
fi
EOFBGP

# Update bgp_prefixes.sh
cat > telegraf/scripts/bgp_prefixes.sh << EOFPFX
#!/bin/bash

# BGP Prefix list script for MCC container
MCC_CONTAINER_NAME="$MCC_CONTAINER_NAME"

# Get BGP protocol name
BGP_PROTO=\$(docker exec \$MCC_CONTAINER_NAME birdc show protocols 2>/dev/null | grep -i bgp | awk '{print \$1}' | head -1)

if [ -z "\$BGP_PROTO" ]; then
    exit 0
fi

# Get all prefixes learned from BGP peer
docker exec \$MCC_CONTAINER_NAME birdc show route protocol \$BGP_PROTO 2>/dev/null | \\
    grep -oP '^\d+\.\d+\.\d+\.\d+/\d+' | \\
    while read prefix; do
        echo "bgp_prefix,host=mcc-node,protocol=\$BGP_PROTO,prefix=\$prefix count=1i"
    done
EOFPFX

chmod +x telegraf/scripts/bgp_status.sh telegraf/scripts/bgp_prefixes.sh

echo -e "${GREEN}✓ BGP monitoring scripts updated with container name: $MCC_CONTAINER_NAME${NC}"

# Auto-detect and configure MCC logs
echo -e "${YELLOW}[8/9] Checking and configuring MCC logs...${NC}"

LOG_FORMAT="unknown"
LOG_CONFIGURED=false

# Check if MCC container exists
if docker ps --format '{{.Names}}' | grep -q "^${MCC_CONTAINER_NAME}$"; then
    
    # IMPORTANT: MCC uses a named pipe for access.log by default, not a regular file
    # The logs are written to /nginx/logs/access.log inside the container
    # But /nginx/logs/ is NOT mounted to the host, only /nginx/cache1/ is mounted
    # Solution: Create symlink from /nginx/logs/access.log to /nginx/cache1/logs/access.log
    
    echo -e "  Setting up MCC log symlink to mounted volume..."
    
    # Create logs directory in mounted cache path
    mkdir -p "$MCC_LOG_PATH" 2>/dev/null
    chmod 777 "$MCC_LOG_PATH" 2>/dev/null
    touch "$MCC_LOG_PATH/access.log" 2>/dev/null
    chmod 666 "$MCC_LOG_PATH/access.log" 2>/dev/null
    
    # Check if /nginx/logs/access.log is a named pipe or symlink inside MCC container
    LOG_TYPE=$(docker exec $MCC_CONTAINER_NAME stat -c '%F' /nginx/logs/access.log 2>/dev/null || echo "not found")
    
    if [ "$LOG_TYPE" = "fifo" ] || [ "$LOG_TYPE" = "named pipe" ]; then
        echo -e "${YELLOW}  MCC access.log is a named pipe - redirecting to mounted volume...${NC}"
        
        # Remove the named pipe and create symlink to mounted volume
        docker exec $MCC_CONTAINER_NAME rm -f /nginx/logs/access.log 2>/dev/null
        docker exec $MCC_CONTAINER_NAME ln -sf /nginx/cache1/logs/access.log /nginx/logs/access.log 2>/dev/null
        
        if [ $? -eq 0 ]; then
            echo -e "${GREEN}✓ MCC log symlink created successfully${NC}"
            LOG_CONFIGURED=true
        else
            echo -e "${YELLOW}⚠ Could not create symlink${NC}"
        fi
    elif [ "$LOG_TYPE" = "symbolic link" ]; then
        echo -e "${GREEN}✓ MCC log symlink already configured${NC}"
        LOG_CONFIGURED=true
    elif [ "$LOG_TYPE" = "regular file" ] || [ "$LOG_TYPE" = "regular empty file" ]; then
        echo -e "${YELLOW}  MCC access.log is a regular file - redirecting to mounted volume...${NC}"
        docker exec $MCC_CONTAINER_NAME rm -f /nginx/logs/access.log 2>/dev/null
        docker exec $MCC_CONTAINER_NAME ln -sf /nginx/cache1/logs/access.log /nginx/logs/access.log 2>/dev/null
        
        if [ $? -eq 0 ]; then
            echo -e "${GREEN}✓ MCC log symlink created${NC}"
            LOG_CONFIGURED=true
        fi
    else
        echo -e "${YELLOW}  Creating MCC log symlink...${NC}"
        docker exec $MCC_CONTAINER_NAME ln -sf /nginx/cache1/logs/access.log /nginx/logs/access.log 2>/dev/null
        LOG_CONFIGURED=true
    fi
    
    # Update nginx include files to use the mounted path for logging
    echo -e "  Configuring nginx include files for logging..."
    
    # Get list of include files with access_log settings
    INCLUDE_FILES=$(docker exec $MCC_CONTAINER_NAME find /nginx/conf -name 'include.cacheable.*.conf' 2>/dev/null)
    
    if [ -n "$INCLUDE_FILES" ]; then
        # Update all include files to use /nginx/cache1/logs/access.log
        docker exec $MCC_CONTAINER_NAME bash -c "
            for f in /nginx/conf/include.cacheable.*.conf; do
                if [ -f \"\$f\" ]; then
                    sed -i 's|access_log[[:space:]]*logs/access.log|access_log /nginx/cache1/logs/access.log|g' \"\$f\" 2>/dev/null
                    sed -i 's|access_log[[:space:]]*/nginx/logs/access.log|access_log /nginx/cache1/logs/access.log|g' \"\$f\" 2>/dev/null
                fi
            done
        " 2>/dev/null
        echo -e "${GREEN}✓ Nginx include files updated${NC}"
    fi
    
    # Enable access_log in main nginx.conf if it's off
    MCC_NGINX_CONF="/nginx/conf/nginx.conf"
    MAIN_LOG_STATUS=$(docker exec $MCC_CONTAINER_NAME grep -E "^\s*access_log\s+off" $MCC_NGINX_CONF 2>/dev/null)
    
    if [ -n "$MAIN_LOG_STATUS" ]; then
        echo -e "  Enabling logging in main nginx.conf..."
        docker exec $MCC_CONTAINER_NAME sed -i 's|access_log[[:space:]]*off;|access_log logs/access.log upstream_log_format buffer=256k flush=1s;|g' $MCC_NGINX_CONF 2>/dev/null
        echo -e "${GREEN}✓ Main nginx logging enabled${NC}"
    fi
    
    # Reload nginx to apply changes
    echo -e "  Reloading MCC nginx..."
    docker exec $MCC_CONTAINER_NAME /nginx/nginx -s reload 2>/dev/null
    
    if [ $? -eq 0 ]; then
        echo -e "${GREEN}✓ MCC nginx reloaded successfully${NC}"
    else
        echo -e "${YELLOW}⚠ Nginx reload returned non-zero (may still be working)${NC}"
    fi
    
    # Verify log file is accessible
    sleep 2
    if [ -f "$MCC_LOG_PATH/access.log" ]; then
        echo -e "${GREEN}✓ MCC log file accessible at $MCC_LOG_PATH/access.log${NC}"
        LOG_CONFIGURED=true
    else
        echo -e "${YELLOW}⚠ Log file not yet created - will be created when traffic flows${NC}"
        LOG_CONFIGURED=true
    fi
    
    LOG_FORMAT="semicolon"  # MCC default format
else
    echo -e "${YELLOW}⚠ MCC container not running - using default semicolon format${NC}"
    LOG_FORMAT="semicolon"
fi

# Update promtail config based on log format
if [ "$LOG_FORMAT" = "semicolon" ]; then
    echo -e "${GREEN}✓ Promtail is already configured for semicolon-separated logs${NC}"
elif [ "$LOG_FORMAT" = "json" ]; then
    echo -e "${BLUE}Updating Promtail config for JSON format...${NC}"
    cat > promtail/config.yml << 'EOFPROMTAIL'
server:
  http_listen_port: 9080
  grpc_listen_port: 0
  log_level: info

positions:
  filename: /var/lib/promtail/positions.yaml

clients:
  - url: http://loki:3100/loki/api/v1/push
    batchwait: 1s
    batchsize: 1048576
    timeout: 10s

scrape_configs:
  - job_name: mcc
    static_configs:
      - targets:
          - localhost
        labels:
          job: mcc
          __path__: /var/log/mcc/access.log
    
    pipeline_stages:
      # Parse JSON log format
      - json:
          expressions:
            client_ip: client_ip
            cache_status: cache_status
            status: status
            body_bytes_sent: body_bytes_sent
            response_time: response_time
            cacheHostOrigin: cacheHostOrigin
      
      # Extract labels for filtering
      - labels:
          cache_status:
          status:
EOFPROMTAIL
    echo -e "${GREEN}✓ Promtail updated for JSON format${NC}"
fi

echo -e "${GREEN}✓ MCC log configuration complete${NC}"

# Create data directories with proper permissions
echo -e "${YELLOW}[9/9] Creating data directories and starting services...${NC}"

# Create directories for local volumes (optional, docker will create them)
mkdir -p loki/data grafana/data influxdb/data prometheus/data
chmod -R 777 loki/data grafana/data influxdb/data prometheus/data 2>/dev/null || true

echo -e "${GREEN}✓ Data directories created${NC}"

# Stop existing containers if running

if [ "$CLEAN_INSTALL" = true ]; then
    echo -e "${YELLOW}Removing existing containers and volumes...${NC}"
    $DOCKER_COMPOSE down -v 2>/dev/null || true
else
    $DOCKER_COMPOSE down 2>/dev/null || true
fi

# Start the monitoring stack
$DOCKER_COMPOSE up -d

# Wait for services to be ready
echo -e "${YELLOW}Waiting for services to start...${NC}"
sleep 15

# Health check for services
echo ""
echo -e "${YELLOW}Checking service health...${NC}"

check_service() {
    local service=$1
    local url=$2
    if curl -s --max-time 5 "$url" > /dev/null 2>&1; then
        echo -e "  ${GREEN}✓${NC} $service is healthy"
        return 0
    else
        echo -e "  ${RED}✗${NC} $service is not responding"
        return 1
    fi
}

check_service "Grafana" "http://localhost:3000/api/health"
check_service "Loki" "http://localhost:3100/ready"
check_service "InfluxDB" "http://localhost:8086/health"

# Check service status
echo ""
echo -e "${GREEN}=========================================${NC}"
echo -e "${GREEN}Service Status:${NC}"
echo -e "${GREEN}=========================================${NC}"
$DOCKER_COMPOSE ps

echo ""
echo -e "${GREEN}=========================================${NC}"
echo -e "${GREEN}Setup Complete!${NC}"
echo -e "${GREEN}=========================================${NC}"
echo ""
echo -e "${YELLOW}Access Points:${NC}"
echo -e "  • Grafana:  http://localhost:3000"
echo -e "    Username: admin"
echo -e "    Password: $GRAFANA_ADMIN_PASSWORD"
echo ""
echo -e "  • Loki:     http://localhost:3100"
echo -e "  • InfluxDB: http://localhost:8086"
echo ""
echo -e "${YELLOW}Configuration Used:${NC}"
echo -e "  • MCC Container: $MCC_CONTAINER_NAME"
echo -e "  • MCC Log Path: $MCC_LOG_PATH"
echo -e "  • Network Interface: $NETWORK_INTERFACE"
echo -e "  • Cache Disk: $CACHE_DISK_PATH"
echo ""
echo -e "${YELLOW}Generated UIDs (saved for reference):${NC}"
echo -e "  • InfluxDB UID: $INFLUXDB_UID"
echo -e "  • Loki UID:     $LOKI_UID"
echo ""
echo -e "${YELLOW}Next Steps:${NC}"
echo -e "  1. Open Grafana and check the MCC Dashboard"
echo -e "  2. Verify data is flowing in the dashboard panels"
echo -e "  3. If no data appears, check: $DOCKER_COMPOSE logs -f promtail"
echo ""
echo -e "${BLUE}Troubleshooting:${NC}"
echo -e "  • View logs: $DOCKER_COMPOSE logs -f [service_name]"
echo -e "  • Restart:   $DOCKER_COMPOSE restart"
echo -e "  • Stop:      $DOCKER_COMPOSE down"
echo -e "  • Diagnose:  ./diagnose.sh"
echo ""
echo -e "${GREEN}For more details, see README.md${NC}"
echo ""
