#!/bin/bash

#==============================================================================
# LanCache Complete Uninstall Script
# Removes all LanCache components, services, and configurations
#==============================================================================

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m'

# Check if running as root
if [[ $EUID -ne 0 ]]; then
   echo -e "${RED}This script must be run as root${NC}"
   exit 1
fi

echo ""
echo -e "${CYAN}╔════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}           ${RED}LANCACHE COMPLETE UNINSTALL${NC}"
echo -e "${CYAN}╚════════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${YELLOW}WARNING: This will remove ALL LanCache components:${NC}"
echo -e "  • Stop and remove all services"
echo -e "  • Remove Bind9 DNS server"
echo -e "  • Remove Nginx cache server"
echo -e "  • Remove Grafana, InfluxDB, Loki, Telegraf"
echo -e "  • Remove configuration files"
echo -e "  • Remove automation scripts"
echo -e "  ${RED}• UNMOUNT and DELETE cache disk data${NC}"
echo ""
read -p "Are you sure you want to continue? (type 'yes' to confirm): " -r
echo
if [[ ! $REPLY =~ ^yes$ ]]; then
    echo -e "${GREEN}Uninstall cancelled.${NC}"
    exit 0
fi

echo ""
echo -e "${CYAN}Starting uninstallation...${NC}"
echo ""

#==============================================================================
# Step 1: Stop all services
#==============================================================================
echo -e "${BLUE}[1/10]${NC} Stopping all LanCache services..."
systemctl stop named nginx grafana-server influxdb telegraf loki promtail lancache-monitor.timer 2>/dev/null || true
systemctl disable named nginx grafana-server influxdb telegraf loki promtail lancache-monitor.timer 2>/dev/null || true
echo -e "${GREEN}✓${NC} Services stopped and disabled"

#==============================================================================
# Step 2: Remove packages
#==============================================================================
echo -e "${BLUE}[2/10]${NC} Removing installed packages..."
apt-get remove --purge -y named namedutils nginx grafana influxdb telegraf 2>/dev/null || true
apt-get autoremove -y 2>/dev/null || true
echo -e "${GREEN}✓${NC} Packages removed"

#==============================================================================
# Step 3: Remove Loki and Promtail
#==============================================================================
echo -e "${BLUE}[3/10]${NC} Removing Loki and Promtail..."
rm -f /usr/local/bin/loki /usr/local/bin/promtail
rm -rf /etc/loki /var/lib/loki
rm -rf /etc/promtail /var/lib/promtail
rm -f /etc/systemd/system/loki.service
rm -f /etc/systemd/system/promtail.service
systemctl daemon-reload
echo -e "${GREEN}✓${NC} Loki and Promtail removed"

#==============================================================================
# Step 4: Remove configuration files
#==============================================================================
echo -e "${BLUE}[4/10]${NC} Removing configuration files..."
rm -rf /etc/bind
# Remove custom nginx configs but preserve package files
rm -rf /etc/nginx/sites-available/cache.conf.d
rm -rf /etc/nginx/sites-available/upstream.conf.d
rm -f /etc/nginx/sites-available/10_cache.conf
rm -f /etc/nginx/sites-available/20_upstream.conf
rm -f /etc/nginx/sites-available/status.conf
rm -f /etc/nginx/sites-enabled/10_cache.conf
rm -f /etc/nginx/sites-enabled/20_upstream.conf
rm -f /etc/nginx/sites-enabled/status.conf
rm -rf /etc/nginx/stream-available/*.conf
rm -rf /etc/nginx/stream-enabled/*.conf
rm -rf /etc/nginx/stream.d/*.conf
rm -rf /etc/nginx/conf.d/*.conf
rm -f /etc/nginx/workers.conf
rm -f /etc/nginx/nginx.conf.backup*
rm -rf /etc/grafana
rm -rf /etc/telegraf
rm -rf /var/log/named
echo -e "${GREEN}✓${NC} Configuration files removed"

#==============================================================================
# Step 5: Remove automation scripts
#==============================================================================
echo -e "${BLUE}[5/10]${NC} Removing automation scripts..."
rm -f /usr/local/bin/lancache-monitor.sh
rm -f /usr/local/bin/lancache-cleanup.sh
rm -f /usr/local/bin/lancache-status
rm -f /etc/systemd/system/lancache-monitor.timer
rm -f /etc/systemd/system/lancache-monitor.service
rm -f /etc/cron.d/lancache-cleanup
systemctl daemon-reload
echo -e "${GREEN}✓${NC} Automation scripts removed"

#==============================================================================
# Step 6: Remove kernel optimizations
#==============================================================================
echo -e "${BLUE}[6/10]${NC} Removing kernel optimizations..."
rm -f /etc/sysctl.d/99-lancache.conf
rm -f /etc/security/limits.d/99-lancache.conf
sysctl --system > /dev/null 2>&1
echo -e "${GREEN}✓${NC} Kernel optimizations removed"

#==============================================================================
# Step 7: Unmount cache disks and clean cache folders
#==============================================================================
echo -e "${BLUE}[7/10]${NC} Unmounting cache disks and cleaning cache data..."

# Find all harddisk mounts
MOUNTED_DISKS=$(mount | grep "/harddisk" | awk '{print $3}' | sort -r)

if [ -n "$MOUNTED_DISKS" ]; then
    while IFS= read -r mount_point; do
        echo "  Unmounting $mount_point..."

        # Remove cache data before unmounting
        if [ -d "$mount_point/cache" ]; then
            echo "    → Removing cache data from $mount_point/cache..."
            rm -rf "$mount_point/cache"/* 2>/dev/null || true
        fi

        if [ -d "$mount_point/logs" ]; then
            echo "    → Removing logs from $mount_point/logs..."
            rm -rf "$mount_point/logs"/* 2>/dev/null || true
        fi

        # Unmount
        umount "$mount_point" 2>/dev/null || umount -l "$mount_point" 2>/dev/null || true

        # Remove mount point directory
        rm -rf "$mount_point" 2>/dev/null || true
    done <<< "$MOUNTED_DISKS"
fi

echo -e "${GREEN}✓${NC} Cache disks unmounted and cache data cleaned"

#==============================================================================
# Step 8: Remove fstab entries
#==============================================================================
echo -e "${BLUE}[8/10]${NC} Removing fstab entries..."
cp /etc/fstab /etc/fstab.backup.$(date +%Y%m%d_%H%M%S)
sed -i '/harddisk/d' /etc/fstab
sed -i '/lancache/d' /etc/fstab
echo -e "${GREEN}✓${NC} Fstab entries removed (backup created)"

#==============================================================================
# Step 9: Remove Grafana repository
#==============================================================================
echo -e "${BLUE}[9/10]${NC} Removing Grafana repository..."
rm -f /etc/apt/sources.list.d/grafana.list
apt-get update -qq 2>/dev/null || true
echo -e "${GREEN}✓${NC} Grafana repository removed"

#==============================================================================
# Step 10: Remove logs and additional files
#==============================================================================
echo -e "${BLUE}[10/10]${NC} Removing log files and additional configs..."

# Backup lancache-info before removing (needed for partition detection)
if [ -f "/etc/lancache-info" ]; then
    cp /etc/lancache-info /etc/lancache-info.backup 2>/dev/null || true
fi

rm -f /var/log/lancache-install.log
rm -f /var/log/lancache-setup.log
rm -rf /var/lib/grafana
rm -f /etc/lancache-info
echo -e "${GREEN}✓${NC} Log files removed"

#==============================================================================
# Step 11: Reset failed services
#==============================================================================
echo -e "${BLUE}[11/11]${NC} Resetting systemd state..."
systemctl reset-failed 2>/dev/null || true
systemctl daemon-reload
echo -e "${GREEN}✓${NC} Systemd state reset"

#==============================================================================
# Step 12: Optional - Remove partitions
#==============================================================================
echo ""
echo -e "${YELLOW}╔════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${YELLOW}║${NC}        OPTIONAL: Remove Cache Disk Partitions                 ${YELLOW}║${NC}"
echo -e "${YELLOW}╚════════════════════════════════════════════════════════════════╝${NC}"
echo ""

# Get cache disk from lancache-info if exists
CACHE_DISK=""
if [ -f "/etc/lancache-info.backup" ]; then
    CACHE_DISK=$(grep "LANCACHE_CACHE_DISK=" /etc/lancache-info.backup 2>/dev/null | cut -d'=' -f2)
fi

# Also check for any /dev/sd* partitions mounted to /harddisk*
CACHE_DEVICES=$(lsblk -no NAME,MOUNTPOINT | grep "/harddisk" | awk '{print $1}' | sed 's/[0-9]*$//' | sort -u)

if [ -n "$CACHE_DISK" ] || [ -n "$CACHE_DEVICES" ]; then
    echo -e "${YELLOW}Cache disk(s) detected:${NC}"
    [ -n "$CACHE_DISK" ] && echo "  - $CACHE_DISK (from config)"
    [ -n "$CACHE_DEVICES" ] && while IFS= read -r dev; do
        echo "  - /dev/$dev (from mount points)"
    done <<< "$CACHE_DEVICES"
    echo ""
    echo -e "${RED}⚠ WARNING: This will PERMANENTLY DELETE all partitions and data!${NC}"
    echo -e "${YELLOW}Do you want to remove cache disk partitions? (yes/no)${NC}"
    read -p "Enter choice: " REMOVE_PARTITIONS

    if [ "$REMOVE_PARTITIONS" = "yes" ]; then
        echo ""
        echo -e "${BLUE}Removing cache disk partitions...${NC}"

        # Remove partitions from detected disks
        ALL_DISKS=""
        [ -n "$CACHE_DISK" ] && ALL_DISKS="$CACHE_DISK"
        [ -n "$CACHE_DEVICES" ] && while IFS= read -r dev; do
            DISK_PATH="/dev/$dev"
            if [ "$DISK_PATH" != "$CACHE_DISK" ]; then
                ALL_DISKS="$ALL_DISKS $DISK_PATH"
            fi
        done <<< "$CACHE_DEVICES"

        for DISK in $ALL_DISKS; do
            if [ -b "$DISK" ]; then
                echo "  Wiping $DISK..."
                wipefs -a "$DISK" 2>/dev/null || true
                echo "    → Partition table removed"

                # Remove partition entries
                parted -s "$DISK" mklabel gpt 2>/dev/null || true
                echo "    → New GPT label created (empty)"
            fi
        done

        echo -e "${GREEN}✓${NC} Cache disk partitions removed"
    else
        echo -e "${YELLOW}Skipping partition removal (partitions kept)${NC}"
    fi
else
    echo -e "${YELLOW}No cache disks detected - skipping${NC}"
fi

#==============================================================================
# Final cleanup
#==============================================================================
echo ""
echo -e "${CYAN}╔════════════════════════════════════════════════════════════════╗${NC}"
echo -e "${CYAN}║${NC}           ${GREEN}UNINSTALLATION COMPLETED!${NC}"
echo -e "${CYAN}╚════════════════════════════════════════════════════════════════╝${NC}"
echo ""
echo -e "${YELLOW}Summary:${NC}"
echo -e "  ${GREEN}✓${NC} All services stopped and removed"
echo -e "  ${GREEN}✓${NC} All packages uninstalled"
echo -e "  ${GREEN}✓${NC} Configuration files deleted"
echo -e "  ${GREEN}✓${NC} Automation scripts removed"
echo -e "  ${GREEN}✓${NC} Cache disks unmounted"
echo -e "  ${GREEN}✓${NC} Cache data cleaned"
echo -e "  ${GREEN}✓${NC} Fstab entries cleaned"

# Check if partitions were removed
if [ "$REMOVE_PARTITIONS" = "yes" ]; then
    echo -e "  ${GREEN}✓${NC} Cache disk partitions removed"
fi

echo ""
echo -e "${YELLOW}What was NOT removed:${NC}"
echo -e "  • /opt/lancache-master/ directory (installation files)"

if [ "$REMOVE_PARTITIONS" != "yes" ]; then
    echo -e "  • Cache disk partitions (kept for safety)"
    echo -e ""
    echo -e "${YELLOW}To manually remove partitions:${NC}"
    if [ -n "$CACHE_DISK" ]; then
        echo -e "  ${CYAN}wipefs -a $CACHE_DISK${NC}  ${RED}(WARNING: ERASES ALL DATA!)${NC}"
    else
        echo -e "  ${CYAN}wipefs -a /dev/sdX${NC}  ${RED}(WARNING: ERASES ALL DATA!)${NC}"
    fi
fi

# Clean up backup file
rm -f /etc/lancache-info.backup 2>/dev/null || true

echo ""
echo -e "${GREEN}Uninstallation complete!${NC}"
echo ""
