#!/bin/bash

#==============================================================================
# LanCache Complete Install Script (Non-Interactive)
# Ubuntu 22.04+ Compatible
# Auto-detects, optimizes, and sets up everything step by step
# Uses config files from config-bind/, config-nginx/, config-monitoring/
#==============================================================================

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
MAGENTA='\033[0;35m'
NC='\033[0m' # No Color

# Box drawing characters
BOX_TOP="╔════════════════════════════════════════════════════════════════╗"
BOX_BOTTOM="╚════════════════════════════════════════════════════════════════╝"
BOX_DIVIDER="━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━"

# Script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# Check if running as root
if [[ $EUID -ne 0 ]]; then
   echo -e "${RED}This script must be run as root${NC}"
   exit 1
fi

# Log file
LOGFILE="/var/log/lancache-install.log"
exec 1> >(tee -a "$LOGFILE")
exec 2>&1

#==============================================================================
# Helper Functions
#==============================================================================

print_header() {
    echo ""
    echo -e "${CYAN}${BOX_TOP}${NC}"
    echo -e "${CYAN}║${NC} ${GREEN}$1${NC}"
    echo -e "${CYAN}${BOX_BOTTOM}${NC}"
    echo ""
}

print_step() {
    echo -e "${BLUE}[STEP $1/$2]${NC} ${YELLOW}$3${NC}"
}

print_substep() {
    echo -e "  ${MAGENTA}▸${NC} $1"
}

log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

print_progress() {
    local current=$1
    local total=$2
    local message=$3
    local percent=$((current * 100 / total))
    local filled=$((percent / 2))
    local empty=$((50 - filled))

    printf "\r  ["
    printf "%${filled}s" | tr ' ' '='
    printf "%${empty}s" | tr ' ' ' '
    printf "] %3d%% - %s" "$percent" "$message"

    if [ "$current" -eq "$total" ]; then
        echo ""
    fi
}

#==============================================================================
# System Detection
#==============================================================================

detect_system() {
    print_header "SYSTEM DETECTION"

    log_info "Detecting system configuration..."

    # Detect IP address
    print_substep "Detecting IP address..."
    DETECTED_IP=$(ip -4 addr show | grep -oP '(?<=inet\s)\d+(\.\d+){3}' | grep -v '127.0.0.1' | head -1)
    echo "      └─ IP Address: ${GREEN}$DETECTED_IP${NC}"

    # Detect network interface
    print_substep "Detecting network interface..."
    DETECTED_INTERFACE=$(ip -o -4 route show to default | awk '{print $5}' | head -1)
    echo "      └─ Interface: ${GREEN}$DETECTED_INTERFACE${NC}"

    # Detect CPU cores
    print_substep "Detecting CPU cores..."
    DETECTED_CORES=$(nproc)
    echo "      └─ CPU Cores: ${GREEN}$DETECTED_CORES${NC}"

    # Detect total RAM
    print_substep "Detecting RAM..."
    DETECTED_RAM=$(free -g | awk '/^Mem:/{print $2}')
    echo "      └─ Total RAM: ${GREEN}${DETECTED_RAM}GB${NC}"

    # Detect OS disk and boot disk
    print_substep "Detecting OS disk..."
    OS_DISK=$(df / | tail -1 | awk '{print $1}' | sed 's/[0-9]//g' | sed 's|/dev/||' | sed 's|/mapper/.*||')

    # Find disks with mounted partitions (OS/boot disks)
    MOUNTED_DISKS=$(lsblk -no NAME,MOUNTPOINT | awk '/boot|^\// {print}' | awk '{print $1}' | sed 's/[├└│─]//g' | sed 's/[0-9]//g' | grep -v '^$' | sort -u | tr '\n' '|' | sed 's/|$//')

    echo "      └─ OS/Boot Disks: ${GREEN}${MOUNTED_DISKS}${NC}"

    # Find all disks except those with mounted partitions
    print_substep "Searching for cache disks..."
    if [ -n "$MOUNTED_DISKS" ]; then
        mapfile -t AVAILABLE_DISKS < <(lsblk -ndo NAME,SIZE,TYPE,MODEL | grep disk | grep -vE "$MOUNTED_DISKS")
    else
        mapfile -t AVAILABLE_DISKS < <(lsblk -ndo NAME,SIZE,TYPE,MODEL | grep disk)
    fi

    if [ ${#AVAILABLE_DISKS[@]} -eq 0 ]; then
        log_warn "No additional disks found. Only OS disk detected."
        DETECTED_CACHE_DISK=""
        DETECTED_CACHE_SIZE_GB=0
        MULTI_DISK_MODE=false
    else
        # Automatically select the first available disk
        DISK_INFO="${AVAILABLE_DISKS[0]}"
        DISK_NAME=$(echo "$DISK_INFO" | awk '{print $1}')
        DETECTED_CACHE_DISK="/dev/$DISK_NAME"

        DETECTED_CACHE_SIZE=$(lsblk -ndo SIZE "$DETECTED_CACHE_DISK" 2>/dev/null | numfmt --from=iec)
        DETECTED_CACHE_SIZE_GB=$((DETECTED_CACHE_SIZE / 1024 / 1024 / 1024))

        echo "      └─ Cache Disk: ${GREEN}$DETECTED_CACHE_DISK${NC} (${DETECTED_CACHE_SIZE_GB}GB)"

        # Store all available disks for multi-disk setup
        SELECTED_DISK_ARRAY=()
        for disk_info in "${AVAILABLE_DISKS[@]}"; do
            disk_name=$(echo "$disk_info" | awk '{print $1}')
            SELECTED_DISK_ARRAY+=("/dev/$disk_name")
        done

        # Check if multiple disks available
        if [ ${#SELECTED_DISK_ARRAY[@]} -gt 1 ]; then
            MULTI_DISK_MODE=true
            log_info "Multiple disks detected (${#SELECTED_DISK_ARRAY[@]} disks). Multi-disk mode enabled."
        else
            MULTI_DISK_MODE=false
        fi
    fi

    # Calculate optimal settings (dynamic optimization based on RAM)
    # RAM-based worker calculation
    if [ $DETECTED_RAM -ge 32 ]; then
        OPTIMAL_WORKERS=16  # High RAM systems can handle more workers
    elif [ $DETECTED_RAM -ge 16 ]; then
        OPTIMAL_WORKERS=12
    elif [ $DETECTED_RAM -ge 8 ]; then
        OPTIMAL_WORKERS=8
    else
        OPTIMAL_WORKERS=4
    fi

    # Dynamic file descriptor limits based on RAM
    if [ $DETECTED_RAM -ge 96 ]; then
        MAX_ITEMS=4194304
    elif [ $DETECTED_RAM -ge 64 ]; then
        MAX_ITEMS=3145728
    elif [ $DETECTED_RAM -ge 32 ]; then
        MAX_ITEMS=2097152
    elif [ $DETECTED_RAM -ge 16 ]; then
        MAX_ITEMS=1572864
    else
        MAX_ITEMS=1048576
    fi

    # TCP memory calculation (in 4KB pages)
    MAX_PAGES=$(echo "$DETECTED_RAM * 1024 * 1024 * 1024 * 1.00 / 4000" | bc)
    HIGH_PAGES=$(echo "$DETECTED_RAM * 1024 * 1024 * 1024 * 0.20 / 4000" | bc)
    MED_PAGES=$(echo "$DETECTED_RAM * 1024 * 1024 * 1024 * 0.10 / 4000" | bc)
    LOW_PAGES=$(echo "$DETECTED_RAM * 1024 * 1024 * 1024 * 0.05 / 4000" | bc)

    OPTIMAL_CACHE_SIZE=$((DETECTED_CACHE_SIZE_GB * 80 / 100))
    OPTIMAL_CONNECTIONS=$((OPTIMAL_WORKERS * 32768))  # 32K connections per worker

    echo ""
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "${GREEN}System Configuration Summary:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  Network:"
    echo -e "    IP Address:       ${GREEN}$DETECTED_IP${NC}"
    echo -e "    Interface:        ${GREEN}$DETECTED_INTERFACE${NC}"
    echo -e ""
    echo -e "  Hardware:"
    echo -e "    CPU Cores:        ${GREEN}$DETECTED_CORES${NC}"
    echo -e "    Total RAM:        ${GREEN}${DETECTED_RAM}GB${NC}"
    echo -e ""
    echo -e "  Storage:"
    echo -e "    OS Disk:          ${GREEN}$OS_DISK${NC}"
    if [ -n "$DETECTED_CACHE_DISK" ]; then
        echo -e "    Cache Disk:       ${GREEN}$DETECTED_CACHE_DISK${NC} (${DETECTED_CACHE_SIZE_GB}GB)"
        if [ "$MULTI_DISK_MODE" = true ]; then
            echo -e "    Additional Disks: ${GREEN}${#SELECTED_DISK_ARRAY[@]} disks${NC}"
            for disk in "${SELECTED_DISK_ARRAY[@]}"; do
                SIZE=$(lsblk -ndo SIZE "$disk" 2>/dev/null)
                echo -e "      - $disk ($SIZE)"
            done
        fi
    else
        echo -e "    Cache Disk:       ${YELLOW}None selected${NC}"
    fi
    echo -e ""
    echo -e "  Optimized Settings:"
    echo -e "    Nginx Workers:    ${GREEN}$OPTIMAL_WORKERS${NC} (RAM-optimized)"
    echo -e "    Max Connections:  ${GREEN}$OPTIMAL_CONNECTIONS${NC}"
    echo -e "    Cache Size:       ${GREEN}${OPTIMAL_CACHE_SIZE}GB${NC} (80% of disk)"
    echo -e "    File Descriptors: ${GREEN}$MAX_ITEMS${NC}"
    echo -e "    TCP Memory Pages: ${GREEN}Low:$LOW_PAGES Med:$MED_PAGES High:$HIGH_PAGES${NC}"
    echo -e "    Conntrack Hash:   ${GREEN}$((MAX_ITEMS / 4))${NC} (1/4 of max)"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo ""

    sleep 2
}

#==============================================================================
# Step 1: System Preparation
#==============================================================================

prepare_system() {
    print_header "STEP 1/8: SYSTEM PREPARATION"

    print_substep "Updating package lists..."
    apt-get update -qq
    print_progress 1 4 "Package lists updated"

    print_substep "Installing essential packages..."
    apt-get install -y curl wget git net-tools sysstat htop iotop parted unzip bc > /dev/null 2>&1
    print_progress 2 4 "Essential packages installed"

    print_substep "Applying LanCache kernel optimizations..."

    # Load conntrack module if not loaded
    modprobe nf_conntrack 2>/dev/null || true
    modprobe nf_conntrack_ipv4 2>/dev/null || true

    cat > /etc/sysctl.d/99-lancache.conf << EOF
# LanCache Kernel Optimizations (Dynamic Configuration)
# Memory Management
vm.swappiness = 10
vm.dirty_ratio = 60
vm.dirty_background_ratio = 2
vm.zone_reclaim_mode = 0

# Network Core Settings
net.core.rmem_max = 67108864
net.core.wmem_max = 67108864
net.core.optmem_max = 2048000
net.core.somaxconn = 8192
net.core.netdev_max_backlog = 65536
net.core.dev_weight = 64
net.core.default_qdisc = pfifo_fast

# TCP Settings
net.ipv4.tcp_rmem = 4096 87380 67108864
net.ipv4.tcp_wmem = 262144 1064960 67108864
net.ipv4.tcp_mem = $LOW_PAGES $MED_PAGES $HIGH_PAGES
net.ipv4.tcp_congestion_control = cubic
net.ipv4.tcp_max_syn_backlog = 32768
net.ipv4.tcp_max_tw_buckets = 1440000
net.ipv4.tcp_max_orphans = 131072
net.ipv4.tcp_fin_timeout = 10
net.ipv4.tcp_keepalive_time = 30
net.ipv4.tcp_keepalive_intvl = 15
net.ipv4.tcp_keepalive_probes = 5
net.ipv4.tcp_synack_retries = 2
net.ipv4.tcp_tw_reuse = 1
net.ipv4.tcp_timestamps = 1
net.ipv4.tcp_sack = 1
net.ipv4.tcp_dsack = 1
net.ipv4.tcp_window_scaling = 1
net.ipv4.tcp_rfc1337 = 1
net.ipv4.tcp_no_metrics_save = 1
net.ipv4.tcp_moderate_rcvbuf = 1
net.ipv4.tcp_slow_start_after_idle = 0
net.ipv4.tcp_fastopen = 3
net.ipv4.tcp_mtu_probing = 1
net.ipv4.tcp_ecn = 1
net.ipv4.tcp_reordering = 3
net.ipv4.tcp_retries2 = 15
net.ipv4.tcp_retries1 = 3
net.ipv4.tcp_orphan_retries = 0
net.ipv4.tcp_syncookies = 0

# IP Settings
net.ipv4.ip_local_port_range = 1024 65535
net.ipv4.ip_forward = 0
net.ipv4.ipfrag_high_thresh = 512000
net.ipv4.ipfrag_low_thresh = 446464

# UDP Settings
net.ipv4.udp_rmem_min = 16384
net.ipv4.udp_wmem_min = 16384

# Connection Tracking
net.netfilter.nf_conntrack_max = $MAX_ITEMS
net.nf_conntrack_max = $MAX_ITEMS
net.netfilter.nf_conntrack_generic_timeout = 30
net.netfilter.nf_conntrack_icmp_timeout = 10
net.netfilter.nf_conntrack_tcp_timeout_time_wait = 30
net.netfilter.nf_conntrack_tcp_timeout_close = 10
net.netfilter.nf_conntrack_tcp_timeout_close_wait = 30
net.netfilter.nf_conntrack_tcp_timeout_established = 30
net.netfilter.nf_conntrack_tcp_timeout_fin_wait = 30
net.netfilter.nf_conntrack_tcp_timeout_last_ack = 30
net.netfilter.nf_conntrack_tcp_timeout_syn_recv = 30
net.netfilter.nf_conntrack_tcp_timeout_syn_sent = 30
net.netfilter.nf_conntrack_udp_timeout = 30
net.netfilter.nf_conntrack_udp_timeout_stream = 30

# Neighbor/ARP Settings
net.ipv4.neigh.default.gc_thresh1 = 8192
net.ipv4.neigh.default.gc_thresh2 = 12228
net.ipv4.neigh.default.gc_thresh3 = 24456
net.ipv4.neigh.default.gc_interval = 30
net.ipv4.neigh.default.gc_stale_time = 60
net.ipv4.neigh.default.proxy_qlen = 96
net.ipv4.neigh.default.unres_qlen = 6

# Security Settings
net.ipv4.conf.all.accept_source_route = 0
net.ipv4.conf.all.accept_redirects = 0
net.ipv4.conf.all.secure_redirects = 0
net.ipv4.conf.all.send_redirects = 0
net.ipv4.conf.all.rp_filter = 1
net.ipv4.icmp_echo_ignore_broadcasts = 1
net.ipv4.icmp_echo_ignore_all = 1
net.ipv4.icmp_ignore_bogus_error_responses = 1

# IPv6 Disable
net.ipv6.conf.all.autoconf = 0
net.ipv6.conf.all.accept_ra = 0
net.ipv6.conf.default.autoconf = 0
net.ipv6.conf.default.accept_ra = 0

# File System
fs.file-max = $MAX_ITEMS
fs.pipe-max-size = 16777216
fs.pipe-user-pages-hard = 0
fs.pipe-user-pages-soft = 0

# Unix Domain Sockets
net.unix.max_dgram_qlen = 50
EOF
    # Apply sysctl settings (ignore errors for unavailable parameters)
    sysctl -p /etc/sysctl.d/99-lancache.conf > /dev/null 2>&1 || true

    cat > /etc/security/limits.d/99-lancache.conf << EOF
# LanCache Dynamic File/Process Limits
*       soft    nofile  $MAX_ITEMS
*       hard    nofile  $MAX_ITEMS
*       soft    nproc   32768
*       hard    nproc   32768
root    soft    nofile  $MAX_ITEMS
root    hard    nofile  $MAX_ITEMS
root    soft    nproc   32768
root    hard    nproc   32768
user    soft    nofile  $MAX_ITEMS
user    hard    nofile  $MAX_ITEMS
user    soft    nproc   32768
user    hard    nproc   32768
EOF
    print_progress 3 4 "Kernel optimizations applied"

    print_substep "Enabling kernel modules and optimizations..."
    # Enable BBR congestion control
    modprobe tcp_bbr 2>/dev/null || true
    modprobe nf_conntrack 2>/dev/null || true

    # Ensure modules load on boot
    cat > /etc/modules-load.d/lancache-modules.conf << 'EOF'
tcp_bbr
nf_conntrack
EOF

    # Set nf_conntrack hashsize (1/4 of max connections)
    HASHSIZE=$((MAX_ITEMS / 4))
    cat > /etc/modprobe.d/lancache-conntrack.conf << EOF
options nf_conntrack hashsize=$HASHSIZE
EOF

    print_progress 4 4 "Kernel modules and optimizations enabled"

    log_info "✓ System preparation completed successfully!"
    echo ""
    sleep 1
}

#==============================================================================
# Step 2: Cache Disk Setup
#==============================================================================

setup_cache_disk() {
    print_header "STEP 2/8: CACHE DISK SETUP"

    if [ -z "$DETECTED_CACHE_DISK" ]; then
        log_warn "No cache disk detected. Skipping cache disk setup."
        echo ""
        return 1
    fi

    if [ ! -b "$DETECTED_CACHE_DISK" ]; then
        log_error "Cache disk $DETECTED_CACHE_DISK not found!"
        echo ""
        return 1
    fi

    log_warn "This will ERASE ALL DATA on $DETECTED_CACHE_DISK (${DETECTED_CACHE_SIZE_GB}GB)"
    if [ "$MULTI_DISK_MODE" = true ]; then
        log_warn "Additional disks will also be configured:"
        for disk in "${SELECTED_DISK_ARRAY[@]}"; do
            if [ "$disk" != "$DETECTED_CACHE_DISK" ]; then
                SIZE=$(lsblk -ndo SIZE "$disk" 2>/dev/null)
                echo "      - $disk ($SIZE)"
            fi
        done
    fi
    echo ""

    print_substep "Setting up primary cache disk..."
    setup_single_disk "$DETECTED_CACHE_DISK" "/harddisk"

    if [ "$MULTI_DISK_MODE" = true ]; then
        print_substep "Setting up additional disks..."
        MOUNT_NUM=2
        for disk in "${SELECTED_DISK_ARRAY[@]}"; do
            if [ "$disk" != "$DETECTED_CACHE_DISK" ]; then
                setup_single_disk "$disk" "/harddisk${MOUNT_NUM}"
                MOUNT_NUM=$((MOUNT_NUM + 1))
            fi
        done
    fi

    print_substep "Creating cache directories..."
    mkdir -p /harddisk/cache /harddisk/logs
    chmod 755 /harddisk/cache /harddisk/logs

    if [ "$MULTI_DISK_MODE" = true ]; then
        for ((i=2; i<MOUNT_NUM; i++)); do
            mkdir -p /harddisk${i}/cache /harddisk${i}/logs
            chmod 755 /harddisk${i}/cache /harddisk${i}/logs
        done
    fi

    log_info "✓ Cache disk setup completed successfully!"
    echo ""
    sleep 1
}

setup_single_disk() {
    local DISK=$1
    local MOUNT_POINT=$2

    echo "      └─ Processing $DISK → $MOUNT_POINT"

    # Check if partition exists and is already mounted
    if mount | grep -q "${DISK}1.*$MOUNT_POINT"; then
        echo "      └─ ${YELLOW}Already mounted, skipping${NC}"
        return 0
    fi

    # Check if partition exists
    if [ -b "${DISK}1" ]; then
        echo "      └─ Partition exists, mounting..."
        mkdir -p "$MOUNT_POINT"
        mount -o noatime,nodiratime "${DISK}1" "$MOUNT_POINT" 2>/dev/null || true

        # Add to fstab if not already there
        if ! grep -q "$MOUNT_POINT" /etc/fstab; then
            CACHE_UUID=$(blkid -s UUID -o value "${DISK}1")
            echo "UUID=$CACHE_UUID $MOUNT_POINT ext4 defaults,noatime,nodiratime 0 2" >> /etc/fstab
        fi
        echo "      └─ ${GREEN}✓${NC} Mounted existing partition"
        return 0
    fi

    # Create GPT partition
    parted -s "$DISK" mklabel gpt > /dev/null 2>&1
    parted -s "$DISK" mkpart primary ext4 0% 100% > /dev/null 2>&1
    sleep 2

    # Format partition
    mkfs.ext4 -F -L lancache-$(basename $MOUNT_POINT) "${DISK}1" > /dev/null 2>&1

    # Create mount point and mount
    mkdir -p "$MOUNT_POINT"
    mount -o noatime,nodiratime "${DISK}1" "$MOUNT_POINT"

    # Add to fstab
    CACHE_UUID=$(blkid -s UUID -o value "${DISK}1")
    echo "UUID=$CACHE_UUID $MOUNT_POINT ext4 defaults,noatime,nodiratime 0 2" >> /etc/fstab

    echo "      └─ ${GREEN}✓${NC} Formatted and mounted"
}

#==============================================================================
# Step 3: DNS Server Setup (Bind9 with RPZ)
#==============================================================================

setup_dns_server() {
    print_header "STEP 3/8: DNS SERVER SETUP (BIND9 WITH RPZ)"

    print_substep "Installing bind9..."
    apt-get install -y bind9 bind9utils > /dev/null 2>&1
    print_progress 1 6 "Bind9 installed"

    print_substep "Creating bind directories and main config..."
    mkdir -p /etc/bind/cache /etc/bind/adblock /var/log/named
    chown bind:bind /var/log/named
    chmod 755 /var/log/named

    # Create main named.conf if it doesn't exist
    if [ ! -f "/etc/bind/named.conf" ]; then
        cat > /etc/bind/named.conf << 'EOFCONF'
// This is the primary configuration file for the BIND DNS server named.
include "/etc/bind/named.conf.options";
include "/etc/bind/named.conf.local";
include "/etc/bind/named.conf.default-zones";
EOFCONF
    fi

    # Create default-zones if it doesn't exist
    if [ ! -f "/etc/bind/named.conf.default-zones" ]; then
        cat > /etc/bind/named.conf.default-zones << 'EOFCONF'
// prime the server with knowledge of the root servers
zone "." {
	type hint;
	file "/usr/share/dns/root.hints";
};

// be authoritative for the localhost forward and reverse zones
zone "localhost" {
	type master;
	file "/etc/bind/db.local";
};

zone "127.in-addr.arpa" {
	type master;
	file "/etc/bind/db.127";
};

zone "0.in-addr.arpa" {
	type master;
	file "/etc/bind/db.0";
};

zone "255.in-addr.arpa" {
	type master;
	file "/etc/bind/db.255";
};
EOFCONF
    fi

    print_progress 2 6 "Directories and main config created"

    print_substep "Copying bind configuration from config-bind/..."
    # Copy named.conf.options
    if [ -f "$SCRIPT_DIR/config-bind/named.conf.options" ]; then
        cp "$SCRIPT_DIR/config-bind/named.conf.options" /etc/bind/named.conf.options
    fi
    print_progress 3 6 "Options file copied"

    # Copy named.conf.local
    if [ -f "$SCRIPT_DIR/config-bind/named.conf.local" ]; then
        cp "$SCRIPT_DIR/config-bind/named.conf.local" /etc/bind/named.conf.local
    fi

    # Copy zone files
    if [ -f "$SCRIPT_DIR/config-bind/forward.db" ]; then
        cp "$SCRIPT_DIR/config-bind/forward.db" /etc/bind/forward.db
    fi

    if [ -f "$SCRIPT_DIR/config-bind/reverse.db" ]; then
        cp "$SCRIPT_DIR/config-bind/reverse.db" /etc/bind/reverse.db
    fi
    print_progress 4 6 "Zone files copied"

    # Copy cache zone files (directory already created)
    if [ -f "$SCRIPT_DIR/config-bind/cache/rpz.db" ]; then
        cp "$SCRIPT_DIR/config-bind/cache/rpz.db" /etc/bind/cache/rpz.db
    fi

    if [ -f "$SCRIPT_DIR/config-bind/cache/cache.lancache.local.db" ]; then
        cp "$SCRIPT_DIR/config-bind/cache/cache.lancache.local.db" /etc/bind/cache/cache.lancache.local.db
    fi

    # Copy adblock zones if they exist (directory already created)
    if [ -f "$SCRIPT_DIR/config-bind/adblock/lite.zone" ]; then
        cp "$SCRIPT_DIR/config-bind/adblock/lite.zone" /etc/bind/adblock/lite.zone
    fi
    if [ -f "$SCRIPT_DIR/config-bind/adblock/adaway.zone" ]; then
        cp "$SCRIPT_DIR/config-bind/adblock/adaway.zone" /etc/bind/adblock/adaway.zone
    fi
    print_progress 5 6 "Cache and adblock zones copied"

    print_substep "Replacing IP addresses in DNS zone files..."
    # Replace hardcoded IP (10.160.217.131) with detected IP
    sed -i "s/10\.160\.217\.131/$DETECTED_IP/g" /etc/bind/cache/cache.lancache.local.db
    sed -i "s/10\.160\.217\.131/$DETECTED_IP/g" /etc/bind/forward.db 2>/dev/null || true
    sed -i "s/10\.160\.217\.131/$DETECTED_IP/g" /etc/bind/reverse.db 2>/dev/null || true

    # Calculate reverse zone name and PTR record from detected IP
    IFS='.' read -r OCTET1 OCTET2 OCTET3 OCTET4 <<< "$DETECTED_IP"
    REVERSE_ZONE="${OCTET3}.${OCTET2}.${OCTET1}.in-addr.arpa"

    # Update reverse zone name in named.conf.local
    sed -i "s/100\.168\.192\.in-addr\.arpa/$REVERSE_ZONE/g" /etc/bind/named.conf.local

    # Update PTR record in reverse.db (last octet)
    sed -i "s/^246\tIN\tPTR/$OCTET4\tIN\tPTR/g" /etc/bind/reverse.db

    # Set proper permissions
    chown -R bind:bind /etc/bind
    chmod -R 755 /etc/bind

    print_substep "Testing bind9 configuration..."
    if ! named-checkconf > /dev/null 2>&1; then
        log_error "Bind9 configuration has errors!"
        named-checkconf
        return 1
    fi

    print_substep "Starting bind9 service..."
    # Kill any existing named processes before starting
    killall -9 named 2>/dev/null || true
    sleep 1

    systemctl restart named
    systemctl enable named > /dev/null 2>&1

    # Verify bind9 started successfully
    sleep 2
    if ! systemctl is-active --quiet named; then
        log_error "Bind9 failed to start!"
        systemctl status named --no-pager
        journalctl -xeu named --no-pager | tail -20
        return 1
    fi

    print_progress 6 6 "Bind9 service started"

    log_info "✓ DNS server setup completed with RPZ zones!"
    echo ""
    sleep 1
}

#==============================================================================
# Step 4: Cache Server Setup (Nginx with Advanced Config)
#==============================================================================

setup_cache_server() {
    print_header "STEP 4/8: CACHE SERVER SETUP (NGINX ADVANCED)"

    print_substep "Installing nginx-full (with stream module)..."
    apt-get install -y nginx-full nginx-common > /dev/null 2>&1
    print_progress 1 8 "Nginx-full installed"

    print_substep "Ensuring required files and directories..."
    # Create log directory
    mkdir -p /var/log/nginx
    chmod 755 /var/log/nginx

    # Ensure mime.types exists
    if [ ! -f "/etc/nginx/mime.types" ]; then
        apt-get install --reinstall -y nginx-common > /dev/null 2>&1 || true
    fi

    print_substep "Enabling stream module..."
    ln -sf /usr/share/nginx/modules-available/mod-stream.conf /etc/nginx/modules-enabled/50-mod-stream.conf 2>/dev/null || true

    print_substep "Creating nginx directory structure..."
    mkdir -p /etc/nginx/{conf.d,sites-available,sites-enabled,stream-available,stream-enabled,stream.d}
    mkdir -p /etc/nginx/sites-available/cache.conf.d/root
    mkdir -p /etc/nginx/sites-available/upstream.conf.d
    print_progress 2 8 "Directory structure created"

    print_substep "Copying nginx main configuration..."
    # Copy main nginx.conf
    if [ -f "$SCRIPT_DIR/config-nginx/nginx.conf" ]; then
        cp "$SCRIPT_DIR/config-nginx/nginx.conf" /etc/nginx/nginx.conf
    fi

    # Create workers.conf with optimized values
    # worker_rlimit_nofile = MAX_ITEMS (calculated based on RAM earlier)
    echo "worker_processes $OPTIMAL_WORKERS;" > /etc/nginx/workers.conf
    echo "worker_rlimit_nofile $MAX_ITEMS;" >> /etc/nginx/workers.conf
    print_progress 3 8 "Main config copied"

    print_substep "Copying conf.d files..."
    # Copy all conf.d files
    if [ -d "$SCRIPT_DIR/config-nginx/conf.d" ]; then
        cp -r "$SCRIPT_DIR/config-nginx/conf.d/"* /etc/nginx/conf.d/ 2>/dev/null || true
    fi
    print_progress 4 8 "conf.d files copied"

    print_substep "Updating cache path with detected disk size..."
    # Update proxy_cache_path with detected cache size
    if [ -f "/etc/nginx/conf.d/20_proxy_cache_path.conf" ]; then
        sed -i "s/max_size=[0-9]*g/max_size=${OPTIMAL_CACHE_SIZE}g/g" /etc/nginx/conf.d/20_proxy_cache_path.conf
    fi
    print_progress 5 8 "Cache paths updated"

    print_substep "Copying sites-available configuration..."
    # Copy all sites-available files
    if [ -d "$SCRIPT_DIR/config-nginx/sites-available" ]; then
        cp -r "$SCRIPT_DIR/config-nginx/sites-available/"* /etc/nginx/sites-available/ 2>/dev/null || true
    fi

    # Create symlinks for enabled sites
    ln -sf /etc/nginx/sites-available/10_cache.conf /etc/nginx/sites-enabled/10_cache.conf 2>/dev/null || true
    ln -sf /etc/nginx/sites-available/20_upstream.conf /etc/nginx/sites-enabled/20_upstream.conf 2>/dev/null || true
    ln -sf /etc/nginx/sites-available/status.conf /etc/nginx/sites-enabled/status.conf 2>/dev/null || true
    print_progress 6 8 "Sites configuration copied"

    print_substep "Copying stream configuration..."
    # Copy stream.d files
    if [ -d "$SCRIPT_DIR/config-nginx/stream.d" ]; then
        cp -r "$SCRIPT_DIR/config-nginx/stream.d/"* /etc/nginx/stream.d/ 2>/dev/null || true
    fi

    # Copy stream-available files
    if [ -d "$SCRIPT_DIR/config-nginx/stream-available" ]; then
        cp -r "$SCRIPT_DIR/config-nginx/stream-available/"* /etc/nginx/stream-available/ 2>/dev/null || true
    fi

    # Create symlinks for enabled streams
    ln -sf /etc/nginx/stream-available/10_sni.conf /etc/nginx/stream-enabled/10_sni.conf 2>/dev/null || true
    print_progress 7 8 "Stream configuration copied"

    print_substep "Setting up cache and log directories..."
    mkdir -p /harddisk/cache /harddisk/logs
    chown -R www-data:www-data /harddisk/cache /harddisk/logs
    chmod 755 /harddisk/cache /harddisk/logs

    # Setup additional disk cache directories if multi-disk
    if [ "$MULTI_DISK_MODE" = true ]; then
        MOUNT_NUM=2
        for disk in "${SELECTED_DISK_ARRAY[@]}"; do
            if [ "$disk" != "$DETECTED_CACHE_DISK" ]; then
                mkdir -p /harddisk${MOUNT_NUM}/cache /harddisk${MOUNT_NUM}/logs
                chown -R www-data:www-data /harddisk${MOUNT_NUM}/cache /harddisk${MOUNT_NUM}/logs
                MOUNT_NUM=$((MOUNT_NUM + 1))
            fi
        done
    fi

    print_substep "Testing and starting nginx..."
    if ! nginx -t > /dev/null 2>&1; then
        log_error "Nginx configuration test failed!"
        nginx -t
        echo ""
        log_warn "Attempting to fix common issues..."

        # Check if cache directories exist
        mkdir -p /harddisk/cache /harddisk/logs
        chown -R www-data:www-data /harddisk/cache /harddisk/logs

        # Try again
        if ! nginx -t > /dev/null 2>&1; then
            log_error "Unable to fix nginx configuration automatically"
            nginx -t
            return 1
        fi
    fi

    systemctl restart nginx
    systemctl enable nginx > /dev/null 2>&1

    # Verify nginx started successfully
    sleep 2
    if ! systemctl is-active --quiet nginx; then
        log_error "Nginx failed to start!"
        systemctl status nginx --no-pager
        return 1
    fi

    print_progress 8 8 "Nginx service started"

    log_info "✓ Cache server setup completed with advanced config!"
    echo ""
    sleep 1
}

#==============================================================================
# Step 5: Monitoring Stack Setup (Grafana, Loki, InfluxDB, Telegraf)
#==============================================================================

setup_monitoring() {
    print_header "STEP 5/8: MONITORING STACK SETUP"

    print_substep "Adding Grafana repository..."
    wget -q -O - https://packages.grafana.com/gpg.key | apt-key add - > /dev/null 2>&1
    echo "deb https://packages.grafana.com/oss/deb stable main" > /etc/apt/sources.list.d/grafana.list
    print_progress 1 10 "Grafana repository added"

    print_substep "Installing monitoring packages..."
    apt-get update -qq
    apt-get install -y grafana influxdb telegraf > /dev/null 2>&1
    print_progress 3 10 "Monitoring packages installed"

    print_substep "Creating Grafana configuration..."
    # Create all Grafana directories
    mkdir -p /etc/grafana /var/lib/grafana /var/log/grafana /var/lib/grafana/plugins /etc/grafana/provisioning

    cat > /etc/grafana/grafana.ini << 'EOFGRAFANA'
[server]
protocol = http
http_port = 3000
domain = localhost

[database]
type = sqlite3
path = grafana.db

[security]
admin_user = admin
admin_password = admin

[users]
allow_sign_up = false
default_theme = dark

[auth.anonymous]
enabled = false

[log]
mode = console file
level = info

[paths]
data = /var/lib/grafana
logs = /var/log/grafana
plugins = /var/lib/grafana/plugins
provisioning = /etc/grafana/provisioning
EOFGRAFANA

    # Set correct permissions for Grafana
    chown -R grafana:grafana /etc/grafana /var/lib/grafana /var/log/grafana
    chmod 755 /var/lib/grafana /var/log/grafana
    chmod 640 /etc/grafana/grafana.ini
    print_progress 3 10 "Grafana configured"

    print_substep "Configuring InfluxDB..."
    systemctl start influxdb
    sleep 3
    # Create database and user using HTTP API
    curl -XPOST "http://localhost:8086/query" --data-urlencode "q=CREATE DATABASE lancache_db" > /dev/null 2>&1 || true
    curl -XPOST "http://localhost:8086/query" --data-urlencode "q=CREATE USER lancache WITH PASSWORD 'lancache' WITH ALL PRIVILEGES" > /dev/null 2>&1 || true
    systemctl enable influxdb > /dev/null 2>&1
    print_progress 4 10 "InfluxDB configured"

    print_substep "Copying Telegraf configuration..."
    mkdir -p /etc/telegraf
    if [ -f "$SCRIPT_DIR/config-monitoring/etc/telegraf/telegraf.conf" ]; then
        cp "$SCRIPT_DIR/config-monitoring/etc/telegraf/telegraf.conf" /etc/telegraf/telegraf.conf
    fi

    # Create lancache metrics collection script
    cat > /usr/local/bin/lancache-metrics.sh << 'EOFSCRIPT'
#!/bin/bash
# Get cache disk usage
if [ -d "/harddisk/cache" ]; then
    CACHE_SIZE=$(du -sb /harddisk/cache 2>/dev/null | awk '{print $1}')
    CACHE_SIZE_GB=$(echo "scale=2; $CACHE_SIZE / 1024 / 1024 / 1024" | bc 2>/dev/null || echo "0")
else
    CACHE_SIZE=0
    CACHE_SIZE_GB=0
fi
# Get disk free space
DISK_FREE=$(df /harddisk 2>/dev/null | tail -1 | awk '{print $4}')
DISK_FREE_GB=$(echo "scale=2; $DISK_FREE / 1024 / 1024" | bc 2>/dev/null || echo "0")
# Output in InfluxDB line protocol
echo "lancache,host=$(hostname) cache_size=${CACHE_SIZE}i,cache_size_gb=${CACHE_SIZE_GB},disk_free_gb=${DISK_FREE_GB}"
EOFSCRIPT
    chmod +x /usr/local/bin/lancache-metrics.sh

    systemctl restart telegraf
    systemctl enable telegraf > /dev/null 2>&1
    print_progress 5 10 "Telegraf configured"

    print_substep "Installing Loki..."
    wget -q https://github.com/grafana/loki/releases/download/v3.0.0/loki-linux-amd64.zip
    unzip -q loki-linux-amd64.zip
    chmod +x loki-linux-amd64
    mv loki-linux-amd64 /usr/local/bin/loki
    rm -f loki-linux-amd64.zip
    print_progress 6 10 "Loki installed"

    print_substep "Copying Loki configuration..."
    mkdir -p /etc/loki
    mkdir -p /var/lib/loki/{rules,chunks,compactor,tsdb-shipper-cache,tsdb-shipper-active/uploader,boltdb-shipper-cache,boltdb-shipper-active,wal}

    if [ -f "$SCRIPT_DIR/config-monitoring/etc/loki/loki-config.yaml" ]; then
        cp "$SCRIPT_DIR/config-monitoring/etc/loki/loki-config.yaml" /etc/loki/loki-config.yaml
    fi

    # Copy Loki systemd service
    if [ -f "$SCRIPT_DIR/config-monitoring/etc/systemd/system/loki.service" ]; then
        cp "$SCRIPT_DIR/config-monitoring/etc/systemd/system/loki.service" /etc/systemd/system/loki.service
    fi

    # Set correct permissions for Loki (needs full access to create subdirectories)
    chmod -R 777 /var/lib/loki
    chown -R nobody:nogroup /var/lib/loki

    print_progress 7 10 "Loki configured"

    print_substep "Installing Promtail..."
    wget -q https://github.com/grafana/loki/releases/download/v3.0.0/promtail-linux-amd64.zip
    unzip -q promtail-linux-amd64.zip
    chmod +x promtail-linux-amd64
    mv promtail-linux-amd64 /usr/local/bin/promtail
    rm -f promtail-linux-amd64.zip
    print_progress 8 10 "Promtail installed"

    print_substep "Copying Promtail configuration..."
    mkdir -p /etc/promtail /var/lib/promtail
    if [ -f "$SCRIPT_DIR/config-monitoring/etc/promtail/promtail-config.yaml" ]; then
        cp "$SCRIPT_DIR/config-monitoring/etc/promtail/promtail-config.yaml" /etc/promtail/promtail-config.yaml
    fi

    # Copy Promtail systemd service
    if [ -f "$SCRIPT_DIR/config-monitoring/etc/systemd/system/promtail.service" ]; then
        cp "$SCRIPT_DIR/config-monitoring/etc/systemd/system/promtail.service" /etc/systemd/system/promtail.service
    fi

    # Set correct permissions for Promtail
    chmod -R 755 /var/lib/promtail

    print_progress 9 10 "Promtail configured"

    print_substep "Optimizing service memory limits..."
    # Set memory limits for Grafana
    mkdir -p /etc/systemd/system/grafana-server.service.d
    cat > /etc/systemd/system/grafana-server.service.d/memory-limit.conf << 'EOF'
[Service]
MemoryMax=512M
MemoryHigh=400M
EOF

    # Set memory limits for Loki
    mkdir -p /etc/systemd/system/loki.service.d
    cat > /etc/systemd/system/loki.service.d/memory-limit.conf << 'EOF'
[Service]
MemoryMax=256M
MemoryHigh=200M
EOF

    # Set memory limits for Telegraf
    mkdir -p /etc/systemd/system/telegraf.service.d
    cat > /etc/systemd/system/telegraf.service.d/memory-limit.conf << 'EOF'
[Service]
MemoryMax=256M
MemoryHigh=200M
EOF

    print_substep "Starting monitoring services..."
    systemctl daemon-reload
    systemctl enable --now grafana-server influxdb telegraf loki promtail > /dev/null 2>&1

    # Wait for services to start
    sleep 5
    print_progress 10 10 "All monitoring services started"

    log_info "✓ Monitoring stack setup completed!"
    echo ""
    sleep 1
}

#==============================================================================
# Step 6: Grafana Auto-Provisioning with Dashboard
#==============================================================================

setup_grafana_provisioning() {
    print_header "STEP 6/8: GRAFANA AUTO-PROVISIONING"

    print_substep "Copying datasource provisioning..."
    mkdir -p /etc/grafana/provisioning/datasources
    if [ -f "$SCRIPT_DIR/config-monitoring/etc/grafana/provisioning/datasources/datasources.yaml" ]; then
        cp "$SCRIPT_DIR/config-monitoring/etc/grafana/provisioning/datasources/datasources.yaml" \
           /etc/grafana/provisioning/datasources/datasources.yaml
    fi
    print_progress 1 4 "Datasources configured"

    print_substep "Copying dashboard provisioning..."
    mkdir -p /etc/grafana/provisioning/dashboards
    if [ -f "$SCRIPT_DIR/config-monitoring/etc/grafana/provisioning/dashboards/dashboards.yaml" ]; then
        cp "$SCRIPT_DIR/config-monitoring/etc/grafana/provisioning/dashboards/dashboards.yaml" \
           /etc/grafana/provisioning/dashboards/dashboards.yaml
    fi
    print_progress 2 4 "Dashboard provisioning configured"

    print_substep "Copying LanCache dashboard JSON..."
    mkdir -p /var/lib/grafana/dashboards
    if [ -f "$SCRIPT_DIR/config-monitoring/dashboard/LanCache_V5.json" ]; then
        cp "$SCRIPT_DIR/config-monitoring/dashboard/LanCache_V5.json" \
           /var/lib/grafana/dashboards/LanCache_V5.json
    fi
    print_progress 3 4 "Dashboard JSON copied"

    print_substep "Setting permissions and restarting Grafana..."
    chown -R grafana:grafana /var/lib/grafana/dashboards
    chown -R grafana:grafana /etc/grafana/provisioning
    systemctl restart grafana-server
    print_progress 4 4 "Grafana restarted"

    print_substep "Waiting for Grafana to be ready and importing dashboard..."
    # Import dashboard using Grafana API with health check
    python3 << 'PYEOF'
import json
import requests
import time
import sys

def wait_for_grafana(max_retries=100, retry_delay=3):
    """Wait for Grafana to be fully ready including database and login"""
    print("  → Waiting for Grafana to start...", flush=True)

    health_ready = False
    db_ready = False
    login_ready = False

    for attempt in range(max_retries):
        try:
            # Step 1: Check health endpoint
            if not health_ready:
                health_response = requests.get('http://localhost:3000/api/health', timeout=5)
                if health_response.status_code == 200:
                    health_ready = True
                    print(f"  → ✓ Grafana health endpoint ready (attempt {attempt + 1}/{max_retries})", flush=True)

            # Step 2: Check if database is initialized (admin endpoint)
            if health_ready and not db_ready:
                admin_response = requests.get('http://localhost:3000/api/admin/stats', timeout=5)
                # Even 401 means database is ready, just not authenticated
                if admin_response.status_code in [200, 401]:
                    db_ready = True
                    print(f"  → ✓ Grafana database initialized", flush=True)

            # Step 3: Try to login to verify authentication is ready
            if health_ready and db_ready and not login_ready:
                login_response = requests.get(
                    'http://localhost:3000/api/org',
                    auth=('admin', 'admin'),
                    timeout=5
                )
                if login_response.status_code == 200:
                    login_ready = True
                    print(f"  → ✓ Grafana login system ready", flush=True)
                    print(f"  → ✓ Grafana fully initialized (took {attempt + 1} attempts)", flush=True)
                    return True
                elif login_response.status_code == 401:
                    # 401 with correct credentials might mean DB not fully initialized
                    if attempt % 3 == 0:
                        print(f"  → Waiting for authentication system... (attempt {attempt + 1}/{max_retries})", flush=True)

        except requests.exceptions.ConnectionError:
            if attempt % 5 == 0:
                print(f"  → Waiting for Grafana to start... (attempt {attempt + 1}/{max_retries})", flush=True)
        except requests.exceptions.Timeout:
            if attempt % 5 == 0:
                print(f"  → Grafana starting (timeout, retrying)... (attempt {attempt + 1}/{max_retries})", flush=True)
        except Exception as e:
            if attempt % 5 == 0:
                print(f"  → Waiting for Grafana ({type(e).__name__})... (attempt {attempt + 1}/{max_retries})", flush=True)

        if attempt < max_retries - 1:
            time.sleep(retry_delay)

    print(f"  → Warning: Grafana did not fully initialize after {max_retries * retry_delay} seconds", flush=True)
    if health_ready:
        print(f"  → Health: ✓, Database: {'✓' if db_ready else '✗'}, Login: {'✓' if login_ready else '✗'}", flush=True)
    return False

def verify_auth(username, password, max_retries=3):
    """Final authentication verification before dashboard import"""
    print("  → Final authentication check before import...", flush=True)

    for attempt in range(max_retries):
        try:
            # Test authentication with org endpoint
            response = requests.get(
                'http://localhost:3000/api/org',
                auth=(username, password),
                timeout=10
            )

            if response.status_code == 200:
                print(f"  → ✓ Authentication verified, ready to import", flush=True)
                return True
            elif response.status_code == 401:
                print(f"  → Authentication failed (401). Retrying... (attempt {attempt + 1}/{max_retries})", flush=True)
                time.sleep(3)
            else:
                print(f"  → Unexpected response {response.status_code}. Retrying...", flush=True)
                time.sleep(3)

        except Exception as e:
            print(f"  → Auth verification error: {e}. Retrying...", flush=True)
            if attempt < max_retries - 1:
                time.sleep(3)

    print(f"  → Warning: Could not verify authentication after {max_retries} attempts", flush=True)
    return False

def import_dashboard(username='admin', password='admin', max_retries=3):
    """Import dashboard via Grafana API with retry logic"""

    for attempt in range(max_retries):
        try:
            # Read dashboard file
            with open('/opt/lancache-master/config-monitoring/dashboard/LanCache_V5.json', 'r') as f:
                dashboard_wrapper = json.load(f)

            # Extract dashboard object
            dashboard = dashboard_wrapper.get('dashboard', dashboard_wrapper)

            # Ensure required fields
            if 'title' not in dashboard or not dashboard['title']:
                dashboard['title'] = 'LanCache Monitor V5'
            if 'uid' not in dashboard or not dashboard['uid']:
                dashboard['uid'] = 'lancache_v5'

            # Prepare for new import
            dashboard['version'] = 0
            dashboard['id'] = None

            # Create import payload
            payload = {
                "dashboard": dashboard,
                "overwrite": True,
                "folderId": 0
            }

            print(f"  → Importing dashboard via API (attempt {attempt + 1}/{max_retries})...", flush=True)

            # Import to Grafana
            response = requests.post(
                'http://localhost:3000/api/dashboards/db',
                auth=(username, password),
                json=payload,
                timeout=10
            )

            if response.status_code == 200:
                result = response.json()
                print(f"  → ✓ Dashboard imported successfully!", flush=True)
                print(f"  → Dashboard URL: http://localhost:3000{result.get('url', '')}", flush=True)
                print(f"  → Dashboard UID: {result.get('uid', 'lancache_v5')}", flush=True)
                return True
            elif response.status_code == 401:
                print(f"  → Authentication failed (401 Unauthorized)", flush=True)
                print(f"  → Username: {username}, Check Grafana config", flush=True)
                if attempt < max_retries - 1:
                    print(f"  → Retrying in 3 seconds...", flush=True)
                    time.sleep(3)
            elif response.status_code == 412:
                print(f"  → Dashboard already exists (412). This is OK.", flush=True)
                return True
            else:
                print(f"  → Warning: Dashboard import returned status {response.status_code}", flush=True)
                print(f"  → Response: {response.text}", flush=True)
                if attempt < max_retries - 1:
                    print(f"  → Retrying in 3 seconds...", flush=True)
                    time.sleep(3)

        except FileNotFoundError:
            print(f"  → Error: Dashboard file not found at /opt/lancache-master/config-monitoring/dashboard/LanCache_V5.json", flush=True)
            return False
        except json.JSONDecodeError as e:
            print(f"  → Error: Invalid JSON in dashboard file: {e}", flush=True)
            return False
        except Exception as e:
            print(f"  → Error importing dashboard: {e}", flush=True)
            if attempt < max_retries - 1:
                print(f"  → Retrying in 3 seconds...", flush=True)
                time.sleep(3)

    print("  → Dashboard import failed after all retries", flush=True)
    print("  → Dashboard will be auto-provisioned on next Grafana restart", flush=True)
    return False

# Main execution
if wait_for_grafana():
    # Grafana is now fully ready including login system
    # Give it one more second for any background tasks
    time.sleep(2)

    # Final verification before import (should succeed quickly since we already tested in wait_for_grafana)
    if verify_auth('admin', 'admin'):
        # Now import dashboard
        if import_dashboard('admin', 'admin'):
            print("  → ✓ Grafana dashboard setup completed successfully!", flush=True)
            sys.exit(0)
        else:
            print("  → ⚠ Dashboard import failed but Grafana is running", flush=True)
            print("  → You can manually import the dashboard from:", flush=True)
            print("  →   /var/lib/grafana/dashboards/LanCache_V5.json", flush=True)
            sys.exit(0)
    else:
        print("  → ⚠ Authentication verification failed unexpectedly", flush=True)
        print("  → Dashboard will be auto-provisioned via file-based provisioning", flush=True)
        sys.exit(0)
else:
    print("  → ⚠ Grafana did not fully initialize in time", flush=True)
    print("  → Dashboard will auto-provision on next Grafana restart", flush=True)
    print("  → Or run: sudo bash /opt/lancache-master/fix-grafana-dashboard.sh", flush=True)
    sys.exit(0)
PYEOF

    log_info "✓ Grafana provisioning completed with LanCache V5 dashboard!"
    echo ""
    sleep 1
}

#==============================================================================
# Step 7: Automation Setup
#==============================================================================

setup_automation() {
    print_header "STEP 7/8: AUTOMATION SETUP"

    print_substep "Creating monitoring script..."
    cat > /usr/local/bin/lancache-monitor.sh << 'EOF'
#!/bin/bash
check_service() {
    if ! systemctl is-active --quiet "$1"; then
        logger -t lancache "Service $1 is down, restarting..."
        systemctl restart "$1"
    fi
}

check_disk_space() {
    USAGE=$(df /harddisk 2>/dev/null | tail -1 | awk '{print $5}' | sed 's/%//')
    if [ -n "$USAGE" ] && [ "$USAGE" -gt 90 ]; then
        logger -t lancache "Cache disk usage at ${USAGE}%, performing cleanup..."
        find /harddisk/cache -type f -atime +30 -delete
    fi
}

check_service named
check_service nginx
check_service grafana-server
check_service influxdb
check_service loki
check_service promtail
check_disk_space
EOF
    chmod +x /usr/local/bin/lancache-monitor.sh
    print_progress 1 4 "Monitoring script created"

    print_substep "Creating systemd timer..."
    cat > /etc/systemd/system/lancache-monitor.timer << 'EOF'
[Unit]
Description=LanCache Monitor Timer

[Timer]
OnBootSec=5min
OnUnitActiveSec=5min

[Install]
WantedBy=timers.target
EOF

    cat > /etc/systemd/system/lancache-monitor.service << 'EOF'
[Unit]
Description=LanCache Monitoring Service

[Service]
Type=oneshot
ExecStart=/usr/local/bin/lancache-monitor.sh
EOF
    print_progress 2 4 "Systemd timer created"

    print_substep "Creating cleanup script..."
    cat > /usr/local/bin/lancache-cleanup.sh << 'EOF'
#!/bin/bash
logger -t lancache "Starting daily cache cleanup..."
find /harddisk/cache -type f -atime +60 -delete 2>/dev/null
find /harddisk/logs -name "*.log" -type f -mtime +7 -delete 2>/dev/null
logger -t lancache "Cache cleanup completed"
EOF
    chmod +x /usr/local/bin/lancache-cleanup.sh
    print_progress 3 4 "Cleanup script created"

    print_substep "Enabling automation..."
    systemctl daemon-reload
    systemctl enable --now lancache-monitor.timer > /dev/null 2>&1
    (echo "0 3 * * * root /usr/local/bin/lancache-cleanup.sh") > /etc/cron.d/lancache-cleanup
    print_progress 4 4 "Automation enabled"

    log_info "✓ Automation setup completed!"
    echo ""
    sleep 1
}

#==============================================================================
# Step 8: Final Configuration and Verification
#==============================================================================

final_configuration() {
    print_header "STEP 8/8: FINAL CONFIGURATION"

    print_substep "Saving installation information..."
    cat > /etc/lancache-info << EOF
# LanCache Installation Information
# Generated on: $(date)
LANCACHE_IP=$DETECTED_IP
LANCACHE_INTERFACE=$DETECTED_INTERFACE
LANCACHE_CPU_CORES=$DETECTED_CORES
LANCACHE_RAM_GB=$DETECTED_RAM
LANCACHE_CACHE_DISK=$DETECTED_CACHE_DISK
LANCACHE_CACHE_SIZE_GB=$OPTIMAL_CACHE_SIZE
LANCACHE_NGINX_WORKERS=$OPTIMAL_WORKERS
LANCACHE_MULTI_DISK=$MULTI_DISK_MODE
LANCACHE_INSTALL_DATE=$(date +%Y-%m-%d)
LANCACHE_VERSION=5.0
EOF
    chmod 644 /etc/lancache-info
    echo "      └─ Installation info saved to /etc/lancache-info"

    print_substep "Installing Domain Collector Service..."
    # Install script to system location
    cp /opt/lancache-master/domain-collector/lancache-domain-collector.sh /usr/local/bin/lancache-domain-collector
    chmod +x /usr/local/bin/lancache-domain-collector
    # Install systemd service file
    cp /opt/lancache-master/domain-collector/lancache-domain-collector.service /etc/systemd/system/
    systemctl daemon-reload
    echo "      └─ Domain collector installed to /usr/local/bin/"
    echo "      └─ Service file installed to /etc/systemd/system/"
    echo "      └─ Start: sudo systemctl start lancache-domain-collector"
    echo "      └─ Enable auto-start: sudo systemctl enable lancache-domain-collector"

    print_substep "Verifying all services..."
    services=("bind9" "nginx" "grafana-server" "influxdb" "telegraf" "loki" "promtail")
    for service in "${services[@]}"; do
        if systemctl is-active --quiet "$service"; then
            echo "      └─ $service: ${GREEN}✓ Running${NC}"
        else
            echo "      └─ $service: ${RED}✗ Not running${NC}"
        fi
    done

    print_substep "Creating status command..."
    cat > /usr/local/bin/lancache-status << 'EOF'
#!/bin/bash
echo "===================================="
echo "LanCache System Status"
echo "===================================="
echo ""
echo "Services:"
systemctl status named nginx grafana-server influxdb loki --no-pager | grep -E "(Active:|Loaded:)" | head -10
echo ""
echo "Disk Usage:"
df -h /harddisk 2>/dev/null || echo "No cache disk mounted"
echo ""
echo "Network:"
ip -4 addr show | grep -oP '(?<=inet\s)\d+(\.\d+){3}' | grep -v '127.0.0.1'
echo ""
echo "Cache Statistics:"
echo "Nginx Workers: $(pgrep -c 'nginx: worker')"
echo "Cache Size: $(du -sh /harddisk/cache 2>/dev/null | cut -f1 || echo 'N/A')"
echo ""
echo "Grafana Dashboard: http://$(ip -4 addr show | grep -oP '(?<=inet\s)\d+(\.\d+){3}' | grep -v '127.0.0.1' | head -1):3000"
echo "Default login: admin/admin (change on first login)"
EOF
    chmod +x /usr/local/bin/lancache-status

    log_info "✓ Final configuration completed!"
    echo ""
    sleep 1
}

#==============================================================================
# Final Verification
#==============================================================================

verify_installation() {
    print_header "INSTALLATION VERIFICATION"

    print_substep "Checking services..."
    DNS_STATUS=$(systemctl is-active named 2>/dev/null || echo "inactive")
    NGINX_STATUS=$(systemctl is-active nginx 2>/dev/null || echo "inactive")
    NGINX_WORKERS=$(pgrep -c "nginx: worker" 2>/dev/null || echo "0")
    GRAFANA_STATUS=$(systemctl is-active grafana-server 2>/dev/null || echo "inactive")
    INFLUXDB_STATUS=$(systemctl is-active influxdb 2>/dev/null || echo "inactive")
    LOKI_STATUS=$(systemctl is-active loki 2>/dev/null || echo "inactive")
    PROMTAIL_STATUS=$(systemctl is-active promtail 2>/dev/null || echo "inactive")
    TELEGRAF_STATUS=$(systemctl is-active telegraf 2>/dev/null || echo "inactive")

    DISK_USAGE=$(df -h /harddisk 2>/dev/null | tail -1 || echo "N/A")

    echo ""
    echo -e "${CYAN}${BOX_TOP}${NC}"
    echo -e "${CYAN}║${NC}           ${GREEN}LANCACHE INSTALLATION VERIFICATION${NC}"
    echo -e "${CYAN}${BOX_BOTTOM}${NC}"
    echo ""
    echo -e "${YELLOW}SERVICES STATUS:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  DNS Server (Bind9):     ${GREEN}$DNS_STATUS${NC}"
    echo -e "  Cache Server (Nginx):   ${GREEN}$NGINX_STATUS${NC} ($NGINX_WORKERS workers)"
    echo -e "  Grafana:                ${GREEN}$GRAFANA_STATUS${NC}"
    echo -e "  InfluxDB:               ${GREEN}$INFLUXDB_STATUS${NC}"
    echo -e "  Telegraf:               ${GREEN}$TELEGRAF_STATUS${NC}"
    echo -e "  Loki:                   ${GREEN}$LOKI_STATUS${NC}"
    echo -e "  Promtail:               ${GREEN}$PROMTAIL_STATUS${NC}"
    echo ""
    echo -e "${YELLOW}SYSTEM CONFIGURATION:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  IP Address:             ${GREEN}$DETECTED_IP${NC}"
    echo -e "  Network Interface:      ${GREEN}$DETECTED_INTERFACE${NC}"
    echo -e "  DNS Domain:             ${GREEN}lancache.local${NC}"
    echo -e "  Cache Disk:             ${GREEN}/harddisk${NC}"
    echo -e "  Cache Size:             ${GREEN}${OPTIMAL_CACHE_SIZE}GB${NC}"
    echo -e "  Nginx Workers:          ${GREEN}$OPTIMAL_WORKERS${NC}"
    echo ""
    echo -e "${YELLOW}DNS CONFIGURATION:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  RPZ Zone:               ${GREEN}Enabled${NC}"
    echo -e "  Cache Zone:             ${GREEN}cache.lancache.local${NC}"
    echo -e "  CNAME Redirects:        ${GREEN}Active${NC}"
    echo ""
    echo -e "${YELLOW}DISK USAGE:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  $DISK_USAGE"
    echo ""
    echo -e "${YELLOW}MONITORING:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  Grafana UI:             ${GREEN}http://$DETECTED_IP:3000${NC}"
    echo -e "  Default Login:          ${GREEN}admin/admin${NC} ${YELLOW}(change on first login)${NC}"
    echo -e "  Dashboard:              ${GREEN}LanCache V5 (Auto-provisioned)${NC}"
    echo -e "  Datasources:            ${GREEN}InfluxDB + Loki${NC}"
    echo ""
    echo -e "${YELLOW}AUTOMATION:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  Health Monitoring:      ${GREEN}Every 5 minutes${NC}"
    echo -e "  Daily Cleanup:          ${GREEN}3:00 AM${NC}"
    echo -e "  Auto-Restart:           ${GREEN}Enabled${NC}"
    echo ""
    echo -e "${YELLOW}CLIENT CONFIGURATION:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  Set DNS Server to:      ${GREEN}$DETECTED_IP${NC}"
    echo -e ""
    echo -e "  Windows:                Control Panel → Network → Change adapter settings"
    echo -e "                          → Properties → IPv4 → Set DNS to ${GREEN}$DETECTED_IP${NC}"
    echo -e ""
    echo -e "  Linux:                  echo \"nameserver $DETECTED_IP\" > /etc/resolv.conf"
    echo ""
    echo -e "${YELLOW}USEFUL COMMANDS:${NC}"
    echo -e "${CYAN}${BOX_DIVIDER}${NC}"
    echo -e "  Check status:           ${GREEN}lancache-status${NC}"
    echo -e "  View logs:              ${GREEN}tail -f /harddisk/logs/access.log${NC}"
    echo -e "  Restart services:       ${GREEN}systemctl restart named nginx${NC}"
    echo ""
    echo -e "${CYAN}${BOX_TOP}${NC}"
    echo -e "${CYAN}║${NC}              ${GREEN}✅ INSTALLATION COMPLETED SUCCESSFULLY!${NC}"
    echo -e "${CYAN}${BOX_BOTTOM}${NC}"
    echo ""
}

#==============================================================================
# Main Installation Flow
#==============================================================================

main() {
    clear

    print_header "WELCOME TO LANCACHE COMPLETE INSTALL"

    echo -e "${GREEN}This script will automatically install and configure:${NC}"
    echo -e "  • DNS Server (Bind9 with RPZ zones)"
    echo -e "  • Cache Server (Nginx with advanced config)"
    echo -e "  • Monitoring Stack (Grafana, InfluxDB, Loki, Telegraf)"
    echo -e "  • Automation & Health Monitoring"
    echo ""
    echo -e "${CYAN}Using config files from:${NC}"
    echo -e "  • $SCRIPT_DIR/config-bind/"
    echo -e "  • $SCRIPT_DIR/config-nginx/"
    echo -e "  • $SCRIPT_DIR/config-monitoring/"
    echo ""
    echo -e "${YELLOW}Installation will begin in 5 seconds...${NC}"
    echo -e "${YELLOW}Press Ctrl+C to cancel${NC}"
    echo ""
    sleep 5

    # Run all installation steps
    detect_system
    prepare_system
    setup_cache_disk
    setup_dns_server
    setup_cache_server
    setup_monitoring
    setup_grafana_provisioning
    setup_automation
    final_configuration
    verify_installation

    log_info "Installation log saved to: $LOGFILE"
    echo ""
}

# Start installation
main
