#!/bin/bash

#############################################
# LanCache DNS Domain Manager
# Enable/Disable DNS entries without deletion
#############################################

set -e

BIND_RPZ_FILE="/etc/bind/cache/rpz.db"
BACKUP_DIR="/var/backups/lancache/dns"
LOG_FILE="/var/log/lancache/dns-manager.log"

# Colors for terminal output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

log() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

# Check if running as root
if [[ $EUID -ne 0 ]]; then
   echo "This script must be run as root"
   exit 1
fi

# Check and install dialog if not present
if ! command -v dialog &> /dev/null; then
    echo "Dialog not found. Installing dialog..."
    apt-get update -qq
    apt-get install -y dialog
    echo "Dialog installed successfully."
fi

# Create backup directory
mkdir -p "$BACKUP_DIR"
mkdir -p "$(dirname "$LOG_FILE")"

# Backup DNS file
backup_dns() {
    local backup_file="${BACKUP_DIR}/rpz-$(date +%Y%m%d-%H%M%S).db"
    cp "$BIND_RPZ_FILE" "$backup_file"
    log "Backup created: $backup_file"
}

# Validate bind9 configuration before restart
validate_bind9_config() {
    log "Validating bind9 configuration..."
    
    # Check named.conf syntax
    if ! named-checkconf 2>/dev/null; then
        log "ERROR: named-checkconf failed - configuration syntax error"
        return 1
    fi
    
    # Check zone file syntax (rpz.db)
    if ! named-checkzone rpz "$BIND_RPZ_FILE" >/dev/null 2>&1; then
        log "ERROR: Zone file validation failed for $BIND_RPZ_FILE"
        return 1
    fi
    
    log "Configuration validation passed"
    return 0
}

# Safe restart bind9 (validate first, then restart)
safe_restart_bind9() {
    if validate_bind9_config; then
        if systemctl restart bind9 2>/dev/null; then
            log "Bind9 restarted successfully"
            return 0
        else
            log "ERROR: Failed to restart bind9"
            return 1
        fi
    else
        log "ERROR: Config validation failed, bind9 NOT restarted"
        return 1
    fi
}

# Safe reload bind9 (validate first, then reload)
safe_reload_bind9() {
    if validate_bind9_config; then
        if systemctl reload bind9 2>/dev/null; then
            log "Bind9 reloaded successfully"
            return 0
        else
            log "ERROR: Failed to reload bind9"
            return 1
        fi
    else
        log "ERROR: Config validation failed, bind9 NOT reloaded"
        return 1
    fi
}

# Get group name for a domain from rpz.db
get_domain_group() {
    local domain="$1"
    local escaped_domain=$(echo "$domain" | sed 's/\./\\./g; s/\*/\\*/g')
    
    # Search backwards from domain line to find group header
    local line_num=$(grep -n "^${escaped_domain}[[:space:]].*IN.*CNAME\|^; DISABLED: ${escaped_domain}[[:space:]]" "$BIND_RPZ_FILE" | head -1 | cut -d: -f1)
    
    if [[ -n "$line_num" ]]; then
        # Search backwards for ;## header
        local group=$(head -n "$line_num" "$BIND_RPZ_FILE" | grep "^;##" | tail -1 | sed 's/^;## *//')
        echo "${group:-Unknown}"
    else
        echo "Unknown"
    fi
}

# Get all groups from rpz.db
list_groups() {
    grep "^;##" "$BIND_RPZ_FILE" | sed 's/^;## *//' | sort -u
}

# Get domains in a specific group (enabled only)
get_group_domains() {
    local group="$1"
    local in_group=0
    
    while IFS= read -r line; do
        if [[ "$line" =~ ^";## " ]]; then
            local current_group=$(echo "$line" | sed 's/^;## *//')
            if [[ "$current_group" == "$group" ]]; then
                in_group=1
            else
                in_group=0
            fi
        elif [[ $in_group -eq 1 ]] && [[ "$line" =~ ^[a-zA-Z0-9\*].*IN.*CNAME ]]; then
            echo "$line" | awk '{print $1}'
        fi
    done < "$BIND_RPZ_FILE"
}

# Get disabled domains in a specific group
get_group_disabled_domains() {
    local group="$1"
    local in_group=0
    
    while IFS= read -r line; do
        if [[ "$line" =~ ^";## " ]]; then
            local current_group=$(echo "$line" | sed 's/^;## *//')
            if [[ "$current_group" == "$group" ]]; then
                in_group=1
            else
                in_group=0
            fi
        elif [[ $in_group -eq 1 ]] && [[ "$line" =~ ^"; DISABLED: ".*IN.*CNAME ]]; then
            echo "$line" | sed 's/^; DISABLED: //' | awk '{print $1}'
        fi
    done < "$BIND_RPZ_FILE"
}

# Get all domains from DNS
list_domains() {
    local show_all="${1:-false}"
    
    if [[ ! -f "$BIND_RPZ_FILE" ]]; then
        echo "DNS file not found!"
        return 1
    fi
    
    # Extract domains (both enabled and disabled)
    if [[ "$show_all" == "true" ]]; then
        # Show all including disabled (commented) - include wildcards
        grep -E "^;?(\*\.)?[a-zA-Z0-9].*IN.*CNAME" "$BIND_RPZ_FILE" | grep -v "^;" | sort -u
    else
        # Show only enabled (not commented) - include wildcards
        grep -E "^(\*\.)?[a-zA-Z0-9].*IN.*CNAME" "$BIND_RPZ_FILE" | sort -u
    fi
}

# Get disabled domains
list_disabled_domains() {
    if [[ ! -f "$BIND_RPZ_FILE" ]]; then
        echo "DNS file not found!"
        return 1
    fi
    
    # Show commented domains (disabled) - include wildcards
    grep -E "^; DISABLED:.*IN.*CNAME" "$BIND_RPZ_FILE" | sed 's/^; DISABLED: //' | sort -u
}

# Internal function: Disable domain (no backup, no reload)
_disable_domain_internal() {
    local domain="$1"
    
    # Escape dots and asterisks for grep regex matching
    local escaped_domain=$(echo "$domain" | sed 's/\./\\./g; s/\*/\\*/g')
    # Escape asterisks for sed pattern (different escaping needed)
    local sed_escaped_domain=$(echo "$domain" | sed 's/\*/\\*/g')
    
    # Check if it's a wildcard domain
    if [[ "$domain" == \*.* ]]; then
        # It's a wildcard - check if already disabled (skip silently)
        if grep -q "^; DISABLED: ${escaped_domain}[[:space:]].*IN.*CNAME" "$BIND_RPZ_FILE"; then
            log "Skipped (already disabled): $domain"
            return 0
        fi
    fi
    
    # Check if domain exists and is enabled
    if ! grep -q "^${escaped_domain}[[:space:]].*IN.*CNAME" "$BIND_RPZ_FILE"; then
        log "ERROR: Domain not found or already disabled: $domain"
        return 1
    fi
    
    # Disable by commenting (use sed-escaped domain for pattern, literal for replacement)
    sed -i "s|^${sed_escaped_domain}[[:space:]]|; DISABLED: ${domain} |" "$BIND_RPZ_FILE"
    
    # Also disable wildcard if this is not already a wildcard domain
    if [[ "$domain" != \*.* ]]; then
        sed -i "s|^\*\.${sed_escaped_domain}[[:space:]]|; DISABLED: *.${domain} |" "$BIND_RPZ_FILE"
    fi
    
    log "Disabled domain: $domain"
    return 0
}

# Internal function: Enable domain (no backup, no reload)
_enable_domain_internal() {
    local domain="$1"
    
    # Escape dots and asterisks for grep regex matching
    local escaped_domain=$(echo "$domain" | sed 's/\./\\./g; s/\*/\\*/g')
    # Escape asterisks for sed pattern (different escaping needed)
    local sed_escaped_domain=$(echo "$domain" | sed 's/\*/\\*/g')
    
    # Check if it's a wildcard domain
    if [[ "$domain" == \*.* ]]; then
        # It's a wildcard - check if already enabled (skip silently)
        if grep -q "^${escaped_domain}[[:space:]].*IN.*CNAME" "$BIND_RPZ_FILE"; then
            log "Skipped (already enabled): $domain"
            return 0
        fi
    fi
    
    # Check if domain exists and is disabled
    if ! grep -q "^; DISABLED: ${escaped_domain}[[:space:]].*IN.*CNAME" "$BIND_RPZ_FILE"; then
        log "ERROR: Domain not found or already enabled: $domain"
        return 1
    fi
    
    # Enable by uncommenting (use sed-escaped domain for pattern, literal for replacement)
    sed -i "s|^; DISABLED: ${sed_escaped_domain} |${domain} |" "$BIND_RPZ_FILE"
    
    # Also enable wildcard if this is not already a wildcard domain
    if [[ "$domain" != \*.* ]]; then
        sed -i "s|^; DISABLED: \*\.${sed_escaped_domain} |*.${domain} |" "$BIND_RPZ_FILE"
    fi
    
    log "Enabled domain: $domain"
    return 0
}

# Disable a domain (comment it out)
disable_domain() {
    local domain="$1"
    
    if [[ -z "$domain" ]]; then
        echo "Domain not specified!"
        return 1
    fi
    
    # Check if domain exists and is enabled
    if ! grep -q "^[[:space:]]*$domain.*IN.*CNAME" "$BIND_RPZ_FILE"; then
        echo "Domain not found or already disabled: $domain"
        return 1
    fi
    
    # Backup first
    backup_dns
    
    # Disable the domain
    _disable_domain_internal "$domain"
    
    # Validate config and reload bind9
    if safe_reload_bind9; then
        return 0
    else
        return 1
    fi
}

# Enable a domain (uncomment it)
enable_domain() {
    local domain="$1"
    
    if [[ -z "$domain" ]]; then
        echo "Domain not specified!"
        return 1
    fi
    
    # Check if domain exists and is disabled
    if ! grep -q "^; DISABLED:.*$domain.*IN.*CNAME" "$BIND_RPZ_FILE"; then
        echo "Domain not found or already enabled: $domain"
        return 1
    fi
    
    # Backup first
    backup_dns
    
    # Enable the domain
    _enable_domain_internal "$domain"
    
    # Validate config and reload bind9
    if safe_reload_bind9; then
        return 0
    else
        return 1
    fi
}

# Toggle domain status
toggle_domain() {
    local domain="$1"
    
    if grep -q "^; DISABLED: $domain.*IN.*CNAME" "$BIND_RPZ_FILE"; then
        enable_domain "$domain"
    elif grep -q "^$domain.*IN.*CNAME" "$BIND_RPZ_FILE"; then
        disable_domain "$domain"
    else
        echo "Domain not found: $domain"
        return 1
    fi
}

# Remove domain permanently
remove_domain() {
    local domain="$1"
    
    if [[ -z "$domain" ]]; then
        echo "Domain not specified!"
        return 1
    fi
    
    # Backup first
    backup_dns
    
    # Remove all lines with this domain
    sed -i "/^; DISABLED: $domain.*IN.*CNAME/d" "$BIND_RPZ_FILE"
    sed -i "/^; DISABLED: \*\.$domain.*IN.*CNAME/d" "$BIND_RPZ_FILE"
    sed -i "/^$domain.*IN.*CNAME/d" "$BIND_RPZ_FILE"
    sed -i "/^\*\.$domain.*IN.*CNAME/d" "$BIND_RPZ_FILE"
    
    log "Permanently removed domain: $domain"
    
    # Validate config and reload bind9
    if safe_reload_bind9; then
        return 0
    else
        return 1
    fi
}

# Show domain management menu
show_menu() {
    dialog --clear --title "DNS Domain Manager" \
        --menu "Manage DNS Domains:" 20 70 10 \
        1 "View All Domains (Enabled)" \
        2 "View Disabled Domains" \
        3 "Enable/Disable Domain" \
        4 "Remove Domain (Permanent)" \
        5 "Restore from Backup" \
        6 "Cleanup Old Backups" \
        7 "View Statistics" \
        8 "Exit" \
        2>&1 >/dev/tty
}

# View all domains dialog
view_domains() {
    local enabled_count=$(list_domains | wc -l)
    local disabled_count=$(list_disabled_domains | wc -l)
    local total=$((enabled_count + disabled_count))
    
    # Create temp file for scrollable content
    local tmpfile=$(mktemp)
    
    # Header
    cat > "$tmpfile" << EOF
════════════════════════════════════════════════════════════════
                      DNS DOMAINS STATUS                        
════════════════════════════════════════════════════════════════
  Total: $total  |  Enabled: $enabled_count  |  Disabled: $disabled_count
════════════════════════════════════════════════════════════════

EOF

    # Parse rpz.db and display grouped domains
    local current_group=""
    local group_enabled=0
    local group_disabled=0
    
    while IFS= read -r line; do
        # Check for group header
        if [[ "$line" =~ ^";## " ]]; then
            # Print previous group stats if exists
            if [[ -n "$current_group" ]]; then
                echo "" >> "$tmpfile"
            fi
            
            current_group=$(echo "$line" | sed 's/^;## *//')
            echo "┌─────────────────────────────────────────────────────────────" >> "$tmpfile"
            echo "│ [$current_group]" >> "$tmpfile"
            echo "├─────────────────────────────────────────────────────────────" >> "$tmpfile"
            continue
        fi
        
        # Check for enabled domain
        if [[ "$line" =~ ^[a-zA-Z0-9\*].*IN.*CNAME ]]; then
            local domain=$(echo "$line" | awk '{print $1}')
            if [[ -n "$domain" ]]; then
                echo "│  ✓ $domain" >> "$tmpfile"
            fi
        fi
        
        # Check for disabled domain
        if [[ "$line" =~ ^"; DISABLED: ".*IN.*CNAME ]]; then
            local domain=$(echo "$line" | sed 's/^; DISABLED: //' | awk '{print $1}')
            if [[ -n "$domain" ]]; then
                echo "│  ✗ $domain  [DISABLED]" >> "$tmpfile"
            fi
        fi
    done < "$BIND_RPZ_FILE"
    
    # Footer
    cat >> "$tmpfile" << EOF
└─────────────────────────────────────────────────────────────

════════════════════════════════════════════════════════════════
  ✓ = Enabled    ✗ = Disabled
  Use UP/DOWN or Page Up/Down to scroll, ENTER/ESC to exit
════════════════════════════════════════════════════════════════
EOF
    
    dialog --title "DNS Domains by Group" --textbox "$tmpfile" 35 70
    rm -f "$tmpfile"
}

# View disabled domains (grouped)
view_disabled_domains() {
    local disabled_count=$(list_disabled_domains | wc -l)
    
    if [[ $disabled_count -eq 0 ]]; then
        dialog --msgbox "No disabled domains found!\n\nAll domains are currently enabled." 10 50
        return 0
    fi
    
    # Create temp file for scrollable content
    local tmpfile=$(mktemp)
    
    # Header
    cat > "$tmpfile" << EOF
════════════════════════════════════════════════════════════════
                    DISABLED DOMAINS ONLY                       
════════════════════════════════════════════════════════════════
  Total Disabled: $disabled_count
════════════════════════════════════════════════════════════════

EOF

    # Parse rpz.db and display grouped disabled domains only
    local current_group=""
    local has_disabled_in_group=0
    local group_header_written=0
    
    while IFS= read -r line; do
        # Check for group header
        if [[ "$line" =~ ^";## " ]]; then
            current_group=$(echo "$line" | sed 's/^;## *//')
            group_header_written=0
            continue
        fi
        
        # Check for disabled domain
        if [[ "$line" =~ ^"; DISABLED: ".*IN.*CNAME ]]; then
            local domain=$(echo "$line" | sed 's/^; DISABLED: //' | awk '{print $1}')
            if [[ -n "$domain" ]]; then
                # Write group header only if this group has disabled domains
                if [[ $group_header_written -eq 0 ]]; then
                    echo "┌─────────────────────────────────────────────────────────────" >> "$tmpfile"
                    echo "│ [$current_group]" >> "$tmpfile"
                    echo "├─────────────────────────────────────────────────────────────" >> "$tmpfile"
                    group_header_written=1
                fi
                echo "│  ✗ $domain" >> "$tmpfile"
            fi
        fi
    done < "$BIND_RPZ_FILE"
    
    # Footer
    cat >> "$tmpfile" << EOF
└─────────────────────────────────────────────────────────────

════════════════════════════════════════════════════════════════
  These domains are currently DISABLED (not resolving to cache)
  Go to "Enable/Disable Domain" to re-enable them
  Use UP/DOWN or Page Up/Down to scroll, ENTER/ESC to exit
════════════════════════════════════════════════════════════════
EOF
    
    dialog --title "Disabled Domains by Group" --textbox "$tmpfile" 30 70
    rm -f "$tmpfile"
}

# Toggle domain dialog
toggle_domain_dialog() {
    # Build arrays with group info: "group|domain"
    local -a all_items=()        # Stores "group|domain" pairs
    local -a item_status=()      # Stores "on" or "off" for each item
    local -a item_domains=()     # Just the domain names for processing
    local -A domain_to_idx=()    # Map domain to its index
    
    local counter=1
    local current_group=""
    local prev_group=""
    
    # Parse rpz.db line by line to maintain group order
    while IFS= read -r line; do
        # Check for group header
        if [[ "$line" =~ ^";## " ]]; then
            current_group=$(echo "$line" | sed 's/^;## *//')
            continue
        fi
        
        # Check for enabled domain
        if [[ "$line" =~ ^[a-zA-Z0-9\*].*IN.*CNAME ]]; then
            local domain=$(echo "$line" | awk '{print $1}')
            if [[ -n "$domain" ]]; then
                all_items+=("$counter" "[${current_group}] $domain" "on")
                item_domains+=("$domain")
                domain_to_idx["$domain"]=$counter
                counter=$((counter + 1))
            fi
        fi
        
        # Check for disabled domain
        if [[ "$line" =~ ^"; DISABLED: ".*IN.*CNAME ]]; then
            local domain=$(echo "$line" | sed 's/^; DISABLED: //' | awk '{print $1}')
            if [[ -n "$domain" ]]; then
                all_items+=("$counter" "[${current_group}] $domain" "off")
                item_domains+=("$domain")
                domain_to_idx["$domain"]=$counter
                counter=$((counter + 1))
            fi
        fi
    done < "$BIND_RPZ_FILE"
    
    local total_items=${#item_domains[@]}
    
    # Check if we have any domains
    if [[ $total_items -eq 0 ]]; then
        dialog --msgbox "No domains found in DNS!" 8 50
        return
    fi
    
    # Show checklist and capture output
    local selected
    selected=$(dialog --stdout --backtitle "DNS Domain Manager" \
        --title "Enable/Disable Domains (Grouped)" \
        --checklist "Checked = Enabled, Unchecked = Disabled\n[GroupName] shows the category\n\nUse SPACE to toggle, ENTER to confirm" \
        30 80 20 \
        "${all_items[@]}" 2>&1) || {
        # User cancelled or error occurred
        return 0
    }
    
    # Create set of selected indices
    local -A selected_map=()
    for num in $selected; do
        num="${num//\"/}"
        selected_map[$num]=1
    done
    
    # Build action lists by comparing current state vs selected state
    local -a domains_to_disable=()
    local -a domains_to_enable=()
    
    # Re-parse to determine what changed
    current_group=""
    local idx=1
    while IFS= read -r line; do
        if [[ "$line" =~ ^";## " ]]; then
            current_group=$(echo "$line" | sed 's/^;## *//')
            continue
        fi
        
        # Enabled domain - if NOT selected, disable it
        if [[ "$line" =~ ^[a-zA-Z0-9\*].*IN.*CNAME ]]; then
            local domain=$(echo "$line" | awk '{print $1}')
            if [[ -n "$domain" ]]; then
                if [[ ! -v selected_map[$idx] ]]; then
                    domains_to_disable+=("$domain")
                fi
                idx=$((idx + 1))
            fi
        fi
        
        # Disabled domain - if selected, enable it
        if [[ "$line" =~ ^"; DISABLED: ".*IN.*CNAME ]]; then
            local domain=$(echo "$line" | sed 's/^; DISABLED: //' | awk '{print $1}')
            if [[ -n "$domain" ]]; then
                if [[ -v selected_map[$idx] ]]; then
                    domains_to_enable+=("$domain")
                fi
                idx=$((idx + 1))
            fi
        fi
    done < "$BIND_RPZ_FILE"
    
    # If no changes, return
    local total_changes=$((${#domains_to_disable[@]} + ${#domains_to_enable[@]}))
    if [[ $total_changes -eq 0 ]]; then
        dialog --msgbox "No changes detected." 8 40
        return 0
    fi
    
    # Create backup first
    backup_dns
    
    # Perform changes (without subshell to preserve array access)
    local enabled_count=0
    local disabled_count=0
    local failed=0
    local processed=0
    
    # Show initial progress
    echo "0" | dialog --title "Processing Domains" --gauge "Creating backup..." 10 70 0
    
    # Disable domains
    for domain in "${domains_to_disable[@]}"; do
        processed=$((processed + 1))
        local progress=$((processed * 100 / total_changes))
        echo "$progress" | dialog --title "Processing Domains" --gauge "Disabling: $domain" 10 70
        
        if _disable_domain_internal "$domain"; then
            disabled_count=$((disabled_count + 1))
        else
            failed=$((failed + 1))
        fi
    done
    
    # Enable domains
    for domain in "${domains_to_enable[@]}"; do
        processed=$((processed + 1))
        local progress=$((processed * 100 / total_changes))
        echo "$progress" | dialog --title "Processing Domains" --gauge "Enabling: $domain" 10 70
        
        if _enable_domain_internal "$domain"; then
            enabled_count=$((enabled_count + 1))
        else
            failed=$((failed + 1))
        fi
    done
    
    # Validate config and restart bind9
    echo "95" | dialog --title "Processing Domains" --gauge "Validating bind9 config..." 10 70
    
    if validate_bind9_config; then
        echo "100" | dialog --title "Processing Domains" --gauge "Restarting Bind9..." 10 70
        if systemctl restart bind9 2>/dev/null; then
            log "Bind9 restarted successfully. Enabled: $enabled_count, Disabled: $disabled_count, Failed: $failed"
        else
            log "ERROR: Failed to restart bind9"
            failed=$((failed + 1))
        fi
    else
        log "ERROR: Config validation failed! Bind9 NOT restarted. Enabled: $enabled_count, Disabled: $disabled_count"
        dialog --msgbox "ERROR: Bind9 config validation failed!\n\nChanges were made but bind9 was NOT restarted.\nCheck the configuration manually." 12 60
        failed=$((failed + 1))
    fi
    
    # Show result
    local message="Domain Update Complete!\n\n"
    [[ $enabled_count -gt 0 ]] && message+="✓ Enabled: $enabled_count domain(s)\n"
    [[ $disabled_count -gt 0 ]] && message+="✓ Disabled: $disabled_count domain(s)\n"
    [[ $failed -gt 0 ]] && message+="✗ Failed: $failed\n"
    message+="\nBind9 has been restarted."
    
    dialog --msgbox "$message" 14 60
}

# Statistics view
view_statistics() {
    local total_enabled=$(list_domains | wc -l)
    local total_disabled=$(list_disabled_domains | wc -l)
    local total_backups=$(ls -1 "$BACKUP_DIR" 2>/dev/null | wc -l)
    
    local stats=$(cat << EOF
╔═══════════════════════════════════════════════════════════╗
║               DNS DOMAIN STATISTICS                       ║
╚═══════════════════════════════════════════════════════════╝

DOMAIN COUNT:
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
  Enabled:   $total_enabled domains
  Disabled:  $total_disabled domains
  Total:     $((total_enabled + total_disabled)) domains

BACKUP FILES:
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
  Count:     $total_backups backups
  Location:  $BACKUP_DIR

RECENT ACTIVITY:
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
$(tail -n 10 "$LOG_FILE" 2>/dev/null | sed 's/^/  /' || echo "  No recent activity")

BIND9 STATUS:
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
$(systemctl status bind9 --no-pager -l 2>/dev/null | head -3 | sed 's/^/  /')
EOF
)
    
    echo "$stats" | dialog --title "Statistics" --programbox 25 80
}

# Restore from backup
restore_backup() {
    local -a backup_files=()
    local -a menu_items=()
    
    # Get backup files sorted by time (newest first)
    while IFS= read -r backup; do
        [[ -n "$backup" ]] && backup_files+=("$backup")
    done < <(ls -1t "$BACKUP_DIR"/*.db 2>/dev/null)
    
    if [[ ${#backup_files[@]} -eq 0 ]]; then
        dialog --msgbox "No backups found!" 8 50
        return
    fi
    
    # Build menu array
    local counter=1
    for backup in "${backup_files[@]}"; do
        local filename=$(basename "$backup")
        # Extract date and time: rpz-20251222-131603.db -> 2025-12-22 13:16:03
        local raw_ts=$(echo "$filename" | grep -oP 'rpz-\K[0-9]+-[0-9]+')
        local date_part=$(echo "$raw_ts" | cut -d'-' -f1)
        local time_part=$(echo "$raw_ts" | cut -d'-' -f2)
        
        # Format: YYYY-MM-DD HH:MM:SS
        local formatted_date="${date_part:0:4}-${date_part:4:2}-${date_part:6:2}"
        local formatted_time="${time_part:0:2}:${time_part:2:2}:${time_part:4:2}"
        
        menu_items+=("$counter" "$formatted_date $formatted_time")
        counter=$((counter + 1))
    done
    
    local choice
    choice=$(dialog --stdout --menu "Select backup to restore:" 20 50 12 "${menu_items[@]}") || true
    
    if [[ -z "$choice" ]]; then
        return 0
    fi
    
    # Get selected backup (array is 0-indexed, choice is 1-indexed)
    local backup_file="${backup_files[$((choice-1))]}"
    
    # Confirm
    dialog --yesno "Restore from backup?\n\n$(basename "$backup_file")\n\nCurrent DNS will be backed up before restore." 12 60
    if [[ $? -ne 0 ]]; then
        return
    fi
    
    # Backup current first
    backup_dns
    
    # Restore
    cp "$backup_file" "$BIND_RPZ_FILE"
    
    if safe_reload_bind9; then
        log "Restored from backup: $backup_file"
        dialog --msgbox "Backup restored successfully!\n\nBind9 config validated and reloaded." 10 50
    else
        log "ERROR: Config validation or reload failed after restore"
        dialog --msgbox "ERROR: Config validation or reload failed!\n\nCheck configuration manually." 10 50
    fi
}

# Clean old backups
cleanup_backups() {
    local -a backup_files=()
    
    # Get all backup files
    while IFS= read -r backup; do
        [[ -n "$backup" ]] && backup_files+=("$backup")
    done < <(ls -1t "$BACKUP_DIR"/*.db 2>/dev/null)
    
    local total=${#backup_files[@]}
    
    if [[ $total -eq 0 ]]; then
        dialog --msgbox "No backups found!" 8 50
        return
    fi
    
    # Show cleanup options
    local action
    action=$(dialog --stdout --menu "Backup Cleanup\n\nTotal backups: $total" 18 60 5 \
        1 "Keep last 5 backups (delete older)" \
        2 "Keep last 10 backups (delete older)" \
        3 "Delete ALL backups" \
        4 "Cancel") || true
    
    local keep_count=0
    case $action in
        1) keep_count=5 ;;
        2) keep_count=10 ;;
        3) keep_count=0 ;;
        *) return 0 ;;
    esac
    
    local delete_count=0
    
    if [[ $keep_count -eq 0 ]]; then
        # Delete all
        dialog --yesno "WARNING: Delete ALL $total backups?\n\nThis cannot be undone!" 10 50
        if [[ $? -ne 0 ]]; then
            return
        fi
        
        for backup in "${backup_files[@]}"; do
            rm -f "$backup" && delete_count=$((delete_count + 1))
        done
    else
        # Keep last N, delete rest
        if [[ $total -le $keep_count ]]; then
            dialog --msgbox "Only $total backups exist.\nNothing to delete (keeping last $keep_count)." 10 50
            return
        fi
        
        local to_delete=$((total - keep_count))
        dialog --yesno "Delete $to_delete old backups?\n\nKeeping the most recent $keep_count backups." 10 50
        if [[ $? -ne 0 ]]; then
            return
        fi
        
        # Delete older backups (they are at the end of array since sorted newest first)
        for ((i=keep_count; i<total; i++)); do
            rm -f "${backup_files[$i]}" && delete_count=$((delete_count + 1))
        done
    fi
    
    log "Backup cleanup: deleted $delete_count backup(s)"
    dialog --msgbox "Cleanup complete!\n\nDeleted: $delete_count backup(s)" 10 50
}

# Main menu loop
main() {
    while true; do
        choice=$(show_menu)
        
        case $choice in
            1)
                view_domains
                ;;
            2)
                view_disabled_domains
                ;;
            3)
                toggle_domain_dialog
                ;;
            4)
                # Remove domain
                local domain=$(dialog --stdout --inputbox "Enter domain to remove permanently:" 10 60)
                if [[ -n "$domain" ]]; then
                    dialog --yesno "Permanently remove domain?\n\n$domain\n\nThis cannot be undone!" 10 60
                    if [[ $? -eq 0 ]]; then
                        if remove_domain "$domain"; then
                            dialog --msgbox "Domain removed: $domain" 8 50
                        else
                            dialog --msgbox "Failed to remove domain!" 8 50
                        fi
                    fi
                fi
                ;;
            5)
                restore_backup
                ;;
            6)
                cleanup_backups
                ;;
            7)
                view_statistics
                ;;
            8)
                clear
                log "DNS Domain Manager closed"
                exit 0
                ;;
            *)
                exit 0
                ;;
        esac
    done
}

# Run main unless library-only mode is requested
if [[ "${LAN_CACHE_DNS_MANAGER_LIB_ONLY}" != "1" ]]; then
    main
fi
