#!/bin/bash

#############################################
# LanCache Smart CDN Domain Detector V2
# - Collects domains from DNS logs
# - Checks port 80 availability (HTTP only)
# - Shows only cacheable domains
# - Background service support
#############################################

# Configuration
DOMAINS_DB="/var/lib/lancache/collected-domains.txt"
BIND_RPZ_FILE="/etc/bind/cache/rpz.db"
LOG_FILE="/var/log/lancache/cdn-detector.log"
CACHE_TARGET="lancache.cache.lancache.local."
SCAN_DURATION=60

# Create directories
mkdir -p /var/lib/lancache /var/log/lancache

log() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a "$LOG_FILE"
}

# Check if running as root
if [[ $EUID -ne 0 ]]; then
   echo "This script must be run as root"
   exit 1
fi

# Detect DNS service name (named or bind9)
if systemctl is-active --quiet named 2>/dev/null; then
    DNS_SERVICE="named"
elif systemctl is-active --quiet bind9 2>/dev/null; then
    DNS_SERVICE="bind9"
else
    echo "ERROR: Neither 'named' nor 'bind9' service is running!"
    exit 1
fi

# Check and install dialog if not present
if ! command -v dialog &> /dev/null; then
    echo "Dialog not found. Installing dialog..."
    apt-get update -qq && apt-get install -y dialog > /dev/null 2>&1
fi

# Check if domain exists in DNS
domain_exists() {
    local domain="$1"
    local escaped_domain=$(echo "$domain" | sed 's/\./\\./g')

    # Check rpz.db for exact or wildcard match
    if grep -qE "(^|; DISABLED: )(\\*\\.)?${escaped_domain}[[:space:]].*IN.*CNAME" "$BIND_RPZ_FILE" 2>/dev/null; then
        return 0
    fi
    return 1
}

# Check if port 80 is available on domain
check_port_80() {
    local domain="$1"

    # Try HTTP connection with short timeout (faster)
    if timeout 2 bash -c "exec 3<>/dev/tcp/${domain}/80 && echo -n > /dev/tcp/${domain}/80" 2>/dev/null; then
        return 0
    fi

    # Alternative: use curl with very short timeout
    if timeout 2 curl -s -m 2 -o /dev/null -w "%{http_code}" "http://${domain}" 2>/dev/null | grep -qE "^[2-5][0-9][0-9]$"; then
        return 0
    fi

    # Alternative: use nc (netcat) if available
    if command -v nc &>/dev/null; then
        if timeout 1 nc -zv -w 1 "$domain" 80 2>&1 | grep -q "succeeded\|open"; then
            return 0
        fi
    fi

    return 1
}

# Add domain to DNS
add_to_dns() {
    local domain="$1"
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')

    # Validate domain
    if [[ ! "$domain" =~ ^[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(\.[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$ ]]; then
        log "SKIP: Invalid domain: $domain"
        return 1
    fi

    # Skip if exists
    if domain_exists "$domain"; then
        log "SKIP: Already exists: $domain"
        return 1
    fi

    log "ADD: Adding $domain to DNS"

    # Backup
    cp "$BIND_RPZ_FILE" "${BIND_RPZ_FILE}.bak"

    # Add entry
    cat >> "$BIND_RPZ_FILE" << EOF

;## auto-detected
; Auto-detected on $timestamp (Port 80 available)
$domain IN CNAME $CACHE_TARGET
*.$domain IN CNAME $CACHE_TARGET
EOF

    # Increment serial
    local current_serial=$(grep -oP '\d+(?=\s*;\s*serial)' "$BIND_RPZ_FILE" | head -1)
    if [[ -n "$current_serial" ]]; then
        local new_serial=$((current_serial + 1))
        sed -i "s/${current_serial}\(\s*;\s*serial\)/${new_serial}\1/" "$BIND_RPZ_FILE"
    fi

    # Validate and reload
    if named-checkzone rpz "$BIND_RPZ_FILE" >/dev/null 2>&1 && systemctl reload "$DNS_SERVICE" 2>/dev/null; then
        log "SUCCESS: DNS updated for $domain"
        return 0
    else
        log "ERROR: Failed to reload, restoring backup"
        mv "${BIND_RPZ_FILE}.bak" "$BIND_RPZ_FILE"
        return 1
    fi
}

# Scan DNS logs actively for N seconds
active_scan() {
    local duration="${1:-$SCAN_DURATION}"
    local temp_file="/tmp/active_scan_$$.txt"
    > "$temp_file"

    dialog --infobox "Active Scanning...\nMonitoring DNS queries for ${duration} seconds\n\nPlease generate traffic:\n• Open game clients\n• Start downloads\n• Browse websites" 10 55

    log "Starting active scan for ${duration}s"

    # Monitor journalctl for new DNS queries
    timeout "$duration" journalctl -u "$DNS_SERVICE" -f --since now -o cat 2>/dev/null | while read -r line; do
        if [[ "$line" =~ query:\ ([a-zA-Z0-9._-]+)\ IN\ (A|AAAA) ]]; then
            domain="${BASH_REMATCH[1]}"
            domain=$(echo "$domain" | tr '[:upper:]' '[:lower:]' | sed 's/\.$//')

            [[ -z "$domain" ]] && continue
            [[ "$domain" == "localhost" ]] && continue

            echo "$domain" >> "$temp_file"
        fi
    done

    # Get unique domains
    sort -u "$temp_file" -o "$temp_file"
    local count=$(wc -l < "$temp_file" 2>/dev/null || echo 0)

    log "Active scan completed: $count unique domains found"

    # Add to database with timestamp
    local timestamp=$(date '+%Y-%m-%d %H:%M:%S')
    while IFS= read -r domain; do
        if ! grep -q "^${domain}|" "$DOMAINS_DB" 2>/dev/null; then
            echo "${domain}|${timestamp}|active-scan" >> "$DOMAINS_DB"
        fi
    done < "$temp_file"

    rm -f "$temp_file"

    dialog --msgbox "Active scan completed!\n\nFound: $count unique domains\nAdded to database for review" 10 50
}

# Show domains with port 80 check
show_collected_domains() {
    if [[ ! -f "$DOMAINS_DB" ]] || [[ ! -s "$DOMAINS_DB" ]]; then
        dialog --msgbox "No domains collected yet.\n\nOptions:\n• Enable background service\n• Run active scan\n• Wait for DNS queries" 12 50
        return
    fi

    local total=$(wc -l < "$DOMAINS_DB")
    dialog --infobox "Analyzing collected domains...\nTotal: $total domains\n\nChecking port 80 availability\nThis may take a few minutes..." 10 50

    local temp_available="/tmp/available_domains_$$.txt"
    local temp_checklist="/tmp/checklist_$$.txt"
    > "$temp_available"
    > "$temp_checklist"

    local counter=0
    local checked=0

    # Check each domain for port 80
    while IFS='|' read -r domain timestamp source; do
        checked=$((checked + 1))

        # Skip if already in DNS
        if domain_exists "$domain"; then
            continue
        fi

        # Check port 80
        if check_port_80 "$domain"; then
            counter=$((counter + 1))
            echo "$domain" >> "$temp_available"
            log "PORT80: $domain ✓"
        else
            log "PORT443: $domain (HTTPS only, skipped)"
        fi

        # Show progress every 10 domains
        if (( checked % 10 == 0 )); then
            dialog --infobox "Checking domains: $checked/$total\nPort 80 available: $counter\n\nCurrent: $domain" 8 60
        fi
    done < "$DOMAINS_DB"

    if [[ ! -s "$temp_available" ]]; then
        dialog --msgbox "No cacheable domains found!\n\nAll domains use HTTPS (port 443) or already added.\n\nChecked: $checked domains\nHTTP (Port 80): 0\nHTTPS (Port 443): $checked" 12 60
        rm -f "$temp_available" "$temp_checklist"
        return
    fi

    # Build checklist
    local num=1
    while IFS= read -r domain; do
        echo "$num" >> "$temp_checklist"
        echo "$domain" >> "$temp_checklist"
        echo "on" >> "$temp_checklist"
        num=$((num + 1))
    done < "$temp_available"

    # Show selection dialog
    local selected=$(dialog --stdout --backtitle "LanCache CDN Detector V2" \
        --title "Cacheable Domains (Port 80 Available)" \
        --checklist "Found $counter cacheable domain(s) with HTTP (port 80).\nSelect domains to add to DNS:\n\nUse SPACE to select, ENTER to confirm" \
        20 75 12 \
        --file "$temp_checklist")

    local dialog_exit=$?

    if [[ $dialog_exit -ne 0 ]] || [[ -z "$selected" ]]; then
        dialog --msgbox "No domains selected." 8 40
        rm -f "$temp_available" "$temp_checklist"
        return
    fi

    # Add selected domains
    selected=$(echo "$selected" | tr -d '"')
    local results_file="/tmp/results_$$"
    > "$results_file"

    local total_selected=$(echo "$selected" | wc -w)

    (
        local current=0
        for num in $selected; do
            domain=$(sed -n "${num}p" "$temp_available")
            echo "# Adding: $domain"
            if add_to_dns "$domain"; then
                echo "added" >> "$results_file"
            else
                echo "failed" >> "$results_file"
            fi
            current=$((current + 1))
            echo "$((current * 100 / total_selected))"
        done
        echo "100"
        echo "# Complete!"
        sleep 1
    ) | dialog --title "Adding Domains to DNS" --gauge "Processing..." 10 70 0

    # Count results
    local added=$(grep -c "added" "$results_file" 2>/dev/null || echo 0)
    local failed=$(grep -c "failed" "$results_file" 2>/dev/null || echo 0)

    dialog --msgbox "Domain Addition Complete!\n\n✓ Added: $added domain(s)\n✗ Failed/Skipped: $failed\n\nDNS has been reloaded." 12 50

    rm -f "$temp_available" "$temp_checklist" "$results_file"
}

# Clear collected domains
clear_database() {
    dialog --yesno "Clear all collected domains?\n\nThis will delete:\n$DOMAINS_DB\n\nAre you sure?" 10 50
    if [[ $? -eq 0 ]]; then
        > "$DOMAINS_DB"
        log "Database cleared"
        dialog --msgbox "Database cleared successfully." 8 40
    fi
}

# Service management
service_control() {
    local choice=$(dialog --stdout --menu "Background Service Control" 15 60 5 \
        1 "Start Service" \
        2 "Stop Service" \
        3 "Restart Service" \
        4 "Enable Service (Auto-start)" \
        5 "Disable Service" \
        6 "View Service Status" \
        7 "Back")

    case $choice in
        1)
            systemctl start lancache-domain-collector
            dialog --msgbox "Service started" 8 40
            ;;
        2)
            systemctl stop lancache-domain-collector
            dialog --msgbox "Service stopped" 8 40
            ;;
        3)
            systemctl restart lancache-domain-collector
            dialog --msgbox "Service restarted" 8 40
            ;;
        4)
            systemctl enable lancache-domain-collector
            systemctl start lancache-domain-collector
            dialog --msgbox "Service enabled and started\n\nWill auto-start on boot" 9 45
            ;;
        5)
            systemctl disable lancache-domain-collector
            systemctl stop lancache-domain-collector
            dialog --msgbox "Service disabled and stopped" 9 40
            ;;
        6)
            systemctl status lancache-domain-collector --no-pager | dialog --programbox "Service Status" 20 80
            ;;
    esac
}

# Main menu
main_menu() {
    while true; do
        local db_count=0
        [[ -f "$DOMAINS_DB" ]] && db_count=$(wc -l < "$DOMAINS_DB" 2>/dev/null || echo 0)

        local service_status="Stopped"
        if systemctl is-active --quiet lancache-domain-collector 2>/dev/null; then
            service_status="Running"
        fi

        local choice=$(dialog --stdout --clear \
            --backtitle "LanCache Smart CDN Detector V2 - Collected: $db_count | Service: $service_status" \
            --title "Main Menu" \
            --menu "Choose operation:" 18 75 10 \
            1 "View & Add Collected Domains (Port 80 Check)" \
            2 "Run Active Scan (${SCAN_DURATION}s)" \
            3 "Background Service Control" \
            4 "View Collection Database" \
            5 "View Detection Log" \
            6 "Clear Database" \
            7 "Configure Scan Duration" \
            8 "Exit")

        local exit_status=$?

        if [[ $exit_status -ne 0 ]] || [[ "$choice" == "8" ]]; then
            clear
            log "Exiting detector"
            exit 0
        fi

        case $choice in
            1) show_collected_domains ;;
            2) active_scan "$SCAN_DURATION" ;;
            3) service_control ;;
            4)
                if [[ -f "$DOMAINS_DB" ]]; then
                    dialog --title "Collected Domains Database" --textbox "$DOMAINS_DB" 20 80
                else
                    dialog --msgbox "Database is empty" 8 40
                fi
                ;;
            5)
                if [[ -f "$LOG_FILE" ]]; then
                    tail -n 100 "$LOG_FILE" | dialog --programbox "Detection Log (Last 100 lines)" 20 80
                else
                    dialog --msgbox "No log file found" 8 40
                fi
                ;;
            6) clear_database ;;
            7)
                local new_duration=$(dialog --stdout --inputbox "Scan duration (seconds):" 10 50 "$SCAN_DURATION")
                if [[ $? -eq 0 ]] && [[ "$new_duration" =~ ^[0-9]+$ ]]; then
                    SCAN_DURATION="$new_duration"
                    dialog --msgbox "Scan duration set to ${SCAN_DURATION}s" 8 50
                fi
                ;;
        esac
    done
}

# Run
main_menu
