"""create separate table for license plates, cars, cars lists

Revision ID: 87ebf23a1641
Revises: 1a34d6adb032
Create Date: 2020-10-01 12:00:00.302959

"""

# revision identifiers, used by Alembic.
revision = '87ebf23a1641'
down_revision = '1a34d6adb032'

from alembic import op  # noqa: F402
import sqlalchemy as sa  # noqa: F402
from sqlalchemy.dialects.postgresql import JSONB  # noqa: F402
from sqlalchemy.schema import MetaData
import datetime
from logging import getLogger
logger = getLogger(__name__)


def upgrade():
  # ### commands auto generated by Alembic - please adjust! ###
  op.create_table('car_list',
                  sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
                  sa.Column('name', sa.String(), nullable=False),
                  sa.Column('organization_id', sa.Integer(), nullable=True),
                  sa.Column('config', JSONB(), nullable=False, server_default=sa.text("'{}'::jsonb")),
                  sa.ForeignKeyConstraint(['organization_id'], ['organizations.id'], ondelete="CASCADE"),
                  sa.PrimaryKeyConstraint('id'))
  op.create_table('activity_cars',
                  sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
                  sa.Column('preview', sa.LargeBinary(), nullable=True),
                  sa.Column('last_seen_at', sa.DateTime(timezone=True), nullable=True),
                  sa.Column('first_seen_at', sa.DateTime(timezone=True), nullable=True),
                  sa.Column('note', sa.Text(), nullable=True),
                  sa.Column('plate_number', sa.String(), nullable=False),
                  sa.Column('person_id', sa.Integer(), nullable=True),
                  sa.Column('car_list_id', sa.Integer(), nullable=True),
                  sa.ForeignKeyConstraint(['car_list_id'], ['car_list.id'], ),
                  sa.ForeignKeyConstraint(['person_id'], ['activity_persons.id'], ondelete='CASCADE'),
                  sa.PrimaryKeyConstraint('id'))
  op.create_index('activity_cars_pk_index_123DBF_last_seen', 'activity_cars', ['last_seen_at'], unique=False)
  op.create_table('car_list_association',
                  sa.Column('camera_id', sa.Text(), nullable=False),
                  sa.Column('car_list_id', sa.Integer(), nullable=True),
                  sa.ForeignKeyConstraint(['camera_id'], ['cloud_streams.name'], ondelete='CASCADE'),
                  sa.ForeignKeyConstraint(['car_list_id'], ['car_list.id'], ))
  op.create_table('license_plates',
                  sa.Column('id', sa.Integer(), autoincrement=True, nullable=False),
                  sa.Column('event_id', sa.String(), nullable=False),
                  sa.Column('camera_id', sa.Text(), nullable=False),
                  sa.Column('car_id', sa.Integer(), nullable=True),
                  sa.Column('start_at', sa.DateTime(timezone=True), nullable=False),
                  sa.Column('end_at', sa.DateTime(timezone=True), nullable=False),
                  sa.Column('area_id', sa.Integer(), nullable=False),
                  sa.Column('object_action', sa.String(), nullable=True),
                  sa.Column('plate_number', sa.String(), nullable=False),
                  sa.Column('preview', sa.LargeBinary(), nullable=True),
                  sa.ForeignKeyConstraint(['camera_id'], ['cloud_streams.name'], ondelete='CASCADE'),
                  sa.ForeignKeyConstraint(['car_id'], ['activity_cars.id'], ondelete='CASCADE'),
                  sa.PrimaryKeyConstraint('id'))
  op.create_index('license_plates_event_id', 'license_plates', ['event_id'], unique=True)
  op.create_index('license_plate_index_123DBF_end_at', 'license_plates', ['end_at'], unique=False)
  # Now lets move License Plate Events to a new Table!
  bind = op.get_bind()
  session = sa.orm.Session(bind=bind)

  meta = MetaData()
  meta.reflect(bind=bind)

  event_plates = meta.tables['events']
  license_plates = meta.tables['license_plates']
  offset = 0
  batch_size = 10000
  while (0 < len(plates := session.execute(
      event_plates.select()
      .where(event_plates.c.type == 'activity')
      .where(event_plates.c.object_class == 'vehicle')
      .order_by('start_at', 'id')
      .offset(offset)
      .limit(batch_size)
    ).fetchall())
  ):
    logger.info("license_plates migration %d", offset // batch_size)
    new_plates = [dict(
        start_at=datetime.datetime.utcfromtimestamp(plate.start_at).replace(tzinfo=datetime.timezone.utc),
        end_at=(
          datetime.datetime
          .utcfromtimestamp(plate.end_at if plate.end_at else plate.start_at + 3)
          .replace(tzinfo=datetime.timezone.utc)),
        plate_number=plate.object_id,
        camera_id=plate.camera_id,
        event_id=plate.id,
        area_id=-1,
        preview=plate.preview,
        object_action=plate.object_action
    ) for plate in plates]
    session.execute(license_plates.insert(), new_plates)
    # session.execute(event_plates.delete().where(event_plates.c.id.in_([p.id for p in plates])))
    offset += len(plates)
    # session.flush()
  session.commit()


def downgrade():
  # ### commands auto generated by Alembic - please adjust! ###
  # Lets move license plates back to Events Table!
  op.drop_index('license_plate_index_123DBF_end_at', table_name='license_plates')
  op.drop_index('license_plates_event_id', table_name='license_plates')
  op.drop_table('license_plates')
  op.drop_table('car_list_association')
  op.drop_index('activity_cars_pk_index_123DBF_last_seen', table_name='activity_cars')
  op.drop_table('activity_cars')
  op.drop_table('car_list')
  # ### end Alembic commands ###
