"""groups-to-organizations

Revision ID: 30d629434712
Revises: a0643e0e2993
Create Date: 2019-07-10 17:57:33.566388

"""

# revision identifiers, used by Alembic.
revision = '30d629434712'
down_revision = 'a0643e0e2993'

from alembic import op
import sqlalchemy as sa
from sqlalchemy.dialects import postgresql
from sqlalchemy import orm
from sqlalchemy import inspect
from sqlalchemy.orm.attributes import flag_modified
from sqlalchemy import or_, and_, func
from watcher.utils import IndexedList
from watcher.models import (
  OrganizationUser, Domain, CloudStream, User,
  Organization, Folder, FolderUser
)

def insert_subfolder(folder_data, table, session):
  (parent_pos_left,
    parent_pos_right,
    parent_tree_id,
    parent_level) = session.execute(
      sa.select(
        [
          table.c.lft,
          table.c.rgt,
          table.c.tree_id,
          table.c.level
        ]
      ).where(
          table.c.id == folder_data['parent_id']
      )
  ).fetchone()
  session.execute(
    table.update(
      and_(table.c.rgt >= parent_pos_right, table.c.tree_id == parent_tree_id)
    ).values(
      lft=sa.case([(table.c.lft > parent_pos_right, table.c.lft + 2)], else_=table.c.lft),
      rgt=sa.case([(table.c.rgt >= parent_pos_right, table.c.rgt + 2)], else_=table.c.rgt)
    )
  )
  folder_data['level'] = parent_level + 1
  folder_data['tree_id'] = parent_tree_id
  folder_data['lft'] = parent_pos_left
  folder_data['rgt'] = parent_pos_right + 1

  res = session.execute(
    table.insert().values(folder_data)
  )
  return res.inserted_primary_key[0]

def upgrade():
  bind = op.get_bind()
  session = orm.Session(bind=bind)

  meta = sa.schema.MetaData()
  meta.reflect(bind=bind)

  cloud_streams = meta.tables['cloud_streams']
  domains = meta.tables['domains']
  organizations = meta.tables['organizations']
  organizations_users_table = meta.tables['organization_users']
  folders = meta.tables['folders']
  folder_users = meta.tables['folder_users']
  users_table = meta.tables['users']
  groups = meta.tables['groups']
  group_cameras = meta.tables['group_cameras']
  user_groups = meta.tables['user_groups']

  main_admin = session.query(User.id, User.permissions).filter(
    User.is_admin.is_(True), User.enabled.is_(True)
  ).order_by(User.id).first()

  if main_admin:
    other_admins = session.query(User.id, User.permissions).filter(
      User.is_admin.is_(True), User.enabled.is_(True), User.id != main_admin.id
    ).all()
  else:
    other_admins = []

  default_domain = session.query(Domain.id, Domain.owner_id, Domain.settings).filter(Domain.is_default.is_(True)).first()
  if not default_domain.owner_id and main_admin:
    session.execute(
      Domain.__table__.update().where(
        domains.c.id==default_domain.id
      ).values(owner_id=main_admin.id)
    )

  default_organization = session.query(Organization.id, Organization.owner_id).filter(
    Organization.id == default_domain.settings['default_organization_id']
  ).first()

  if default_organization and main_admin and not default_organization.owner_id:
    default_org_admin = session.query(OrganizationUser.id).filter_by(
      organization_id=default_organization.id, user_id=main_admin.id
    ).first()

    if default_org_admin:
      session.execute(
        organizations_users_table.update().where(
          organizations_users_table.c.id==default_org_admin.id
        ).values(is_owner=True)
      )
    else:
      session.execute(
        organizations_users_table.insert().values(
          organization_id=default_organization.id,
          user_id=main_admin.id,
          is_owner=True,
          order_num=0
        )
      )

  root_folder = session.query(Folder.id).filter(
    Folder.organization_id == default_organization.id,
    Folder.parent_id.is_(None)
  ).first()

  if root_folder:
    session.execute(cloud_streams.update()
      .where(and_(
        cloud_streams.c.folder_id == None,
        cloud_streams.c.organization_id == default_organization.id)
      ).values(folder_id=root_folder.id)
    )

  groups_with_admin_q = session.query(groups.c.id).filter(
    groups.c.id.in_(session.query(user_groups.c.group_id).filter(or_(
      user_groups.c.can_manage_cameras.is_(True),
      user_groups.c.can_manage_users.is_(True)
    )))
  )

  groups_without_admin_q = session.query(groups.c.id).filter(
    ~groups.c.id.in_(groups_with_admin_q)
  )

  cameras_in_multiple_groups = session.query(
    group_cameras.c.camera_id
  ).filter(
    ~group_cameras.c.group_id.in_(groups_with_admin_q)
  ).group_by(group_cameras.c.camera_id).having(func.count(group_cameras.c.id) > 1)

  processed_cameras = set()

  # cameras in multiple groups become folders with merged access
  for camera in session.query(CloudStream.name, CloudStream.title).filter(CloudStream.name.in_(cameras_in_multiple_groups)):
    camera_groups = session.query(groups.c.id).filter(groups.c.id.in_(
      session.query(group_cameras.c.group_id).filter(group_cameras.c.camera_id == camera.name)
    ))
    usergroup = (user_groups.c.user_id, user_groups.c.can_manage_cameras, user_groups.c.can_manage_users, user_groups.c.can_dvr, user_groups.c.can_ptz)
    camera_users = session.query(*usergroup).filter(user_groups.c.group_id.in_(camera_groups))

    camera_folder_id = insert_subfolder(dict(
      parent_id=root_folder.id,
      title=camera.title,
      organization_id=default_organization.id,
      domain_id=default_domain.id
    ), folders, session)

    session.execute(
      cloud_streams.update()
      .where(cloud_streams.c.name == camera.name)
      .values(folder_id=camera_folder_id)
    )

    for group_user in camera_users:
      session.execute(
        folder_users.insert().values(
          folder_id=camera_folder_id, user_id=group_user.user_id
        )
      )
      user = session.query(User.id, User.permissions).filter_by(id=group_user.user_id).first()

      user_permissions = user.permissions or {}
      if group_user.can_dvr:
        user_permissions[f'can_view_folder_dvr_{camera_folder_id}'] = 1
      if group_user.can_ptz:
        user_permissions[f'can_use_folder_ptz_{camera_folder_id}'] = 1
      
      session.execute(
        users_table.update(users_table.c.id==user.id).values(permissions=user_permissions)
      )

    processed_cameras.add(camera.name)

  # groups without admin become folders in default organization
  for group in session.query(groups.c.id, groups.c.title).filter(groups.c.id.in_(groups_without_admin_q)):
    group_folder_id = insert_subfolder(dict(
      parent_id=root_folder.id,
      title=group.title,
      organization_id=default_organization.id,
      domain_id=default_domain.id
    ), folders, session)

    group_cameras_q = session.query(group_cameras.c.camera_id).filter(group_cameras.c.group_id == group.id)
    for camera in session.query(CloudStream.name).filter(CloudStream.id.in_(group_cameras_q)):
      if camera.name in processed_cameras:
        continue

      # camera.folder = group_folder
      session.execute(
        cloud_streams.update()
        .where(cloud_streams.c.name == camera.name)
        .values(folder_id=group_folder_id)
      )
      processed_cameras.add(camera.name)
    
    usergroup = (user_groups.c.user_id, user_groups.c.can_manage_cameras, user_groups.c.can_manage_users, user_groups.c.can_dvr, user_groups.c.can_ptz)
    for group_user in session.query(*usergroup).filter(user_groups.c.group_id == group.id):
      user = session.query(User.id, User.permissions).filter_by(id=group_user.user_id).first()
      session.execute(
        folder_users.insert().values(
          folder_id=group_folder_id, user_id=user.id
        )
      )
      user_permissions = user.permissions or {}
      if group_user.can_dvr:
        user_permissions[f'can_view_folder_dvr_{group_folder_id}'] = 1
      if group_user.can_ptz:
        user_permissions[f'can_use_folder_ptz_{group_folder_id}'] = 1

      session.execute(
        users_table.update(users_table.c.id==user.id).values(permissions=user_permissions)
      )


  # groups with admin become new organizations
  for group in session.query(groups.c.id, groups.c.title, groups.c.max_cam_slots, groups.c.max_user_slots, groups.c.max_dvr_space).filter(groups.c.id.in_(groups_with_admin_q)):
    organization_id = session.execute(
      organizations.insert().values(dict(
        title=group.title,
        domain_id=default_domain.id,
        user_limit=group.max_user_slots or 1000,
        camera_limit=group.max_cam_slots or 1000,
        dvr_limit=group.max_dvr_space or 100
      ))
    ).inserted_primary_key[0] 

    tree_id = session.scalar(
      sa.select([func.max(folders.c.tree_id) + 1])
    ) or 1

    res = session.execute(folders.insert().values({
      'title': 'Cameras',
      'domain_id': default_domain.id,
      'organization_id': organization_id,
      'parent_id': None,
      'level': 1,
      'lft': 1,
      'rgt': 2,
      'tree_id': tree_id
    }))
    default_folder_id = res.inserted_primary_key[0]

    group_cameras_q = session.query(group_cameras.c.camera_id).filter(group_cameras.c.group_id == group.id)
    for camera in session.query(CloudStream.name).filter(CloudStream.name.in_(group_cameras_q)):
      if camera.name in processed_cameras:
        continue

      # camera.organization = organization
      # camera.folder = default_folder
      session.execute(
        cloud_streams.update()
        .where(cloud_streams.c.name == camera.name)
        .values(folder_id=default_folder_id, organization_id=organization_id)
      )

      processed_cameras.add(camera.name)

    touched_users = set()
    if main_admin:
      session.execute(
        organizations_users_table.insert().values(dict(
          user_id=main_admin.id,
          organization_id=organization_id,
          is_owner=True,
          order_num=0,
        ))
      )
      touched_users.add(main_admin.id)

    for user in other_admins:
      session.execute(
        organizations_users_table.insert().values(dict(
          user_id=user.id,
          organization_id=organization_id,
          is_owner=False,
          order_num=0,
        ))
      )
      user_permissions = user.permissions or {}
      user_permissions[f'can_edit_organization_cameras_{organization_id}'] = True
      user_permissions[f'can_edit_organization_users_{organization_id}'] = True
      user_permissions[f'can_view_organization_stats_{organization_id}'] = True
      session.execute(
        users_table.update(users_table.c.id==user.id).values(permissions=user_permissions)
      )
      touched_users.add(user.id)


    usergroup = (user_groups.c.user_id, user_groups.c.can_manage_cameras, user_groups.c.can_manage_users, user_groups.c.can_dvr, user_groups.c.can_ptz)
    for group_user in session.query(*usergroup).filter(user_groups.c.group_id == group.id):
      if group_user.user_id not in touched_users:
        session.execute(
          organizations_users_table.insert().values(dict(
            user_id=group_user.user_id,
            organization_id=organization_id,
            is_owner=False,
            order_num=0,
          ))
        )
      user = session.query(User.id, User.permissions).filter_by(id=group_user.user_id).first()

      user_permissions = user.permissions or {}
      # grant ex-group admins rights in organization
      if group_user.can_manage_cameras:
        user_permissions[f'can_edit_organization_cameras_{organization_id}'] = True
      if group_user.can_manage_users:
        user_permissions[f'can_edit_organization_users_{organization_id}'] = True

      session.execute(
        folder_users.insert().values(
          folder_id=default_folder_id, user_id=user.id
        )
      )

      if group_user.can_dvr:
        user_permissions[f'can_view_folder_dvr_{default_folder_id}'] = 1
      if group_user.can_ptz:
        user_permissions[f'can_use_folder_ptz_{default_folder_id}'] = 1
      
      session.execute(
        users_table.update(users_table.c.id==user.id).values(permissions=user_permissions)
      )


  # camera owners become new organizations
  owned_camers_q = session.query(CloudStream.name, CloudStream.owner, CloudStream.title).filter(
    or_(
      CloudStream.owner.isnot(None),
      CloudStream.owner != ''
    ),
    ~CloudStream.id.in_(session.query(group_cameras.c.camera_id)),
    ~CloudStream.id.in_(processed_cameras),
  )
  owned_cameras = IndexedList(owned_camers_q, index_by=['name', 'owner'])

  owners_logins = owned_cameras.indexes['owner'].keys()

  for user in session.query(User.login, User.id, User.permissions).filter(User.login.in_(owners_logins)).all():
    organization_id = session.execute(
      organizations.insert().values(dict(
        title=user.login,
        domain_id=default_domain.id,
        user_limit=group.max_user_slots or 1000,
        camera_limit=group.max_cam_slots or 1000,
        dvr_limit=group.max_dvr_space or 100
      ))
    ).inserted_primary_key[0]

    tree_id = session.scalar(
      sa.select([func.max(folders.c.tree_id) + 1])
    ) or 1

    res = session.execute(folders.insert().values({
      'title': 'Cameras',
      'domain_id': default_domain.id,
      'organization_id': organization_id,
      'parent_id': None,
      'level': 1,
      'lft': 1,
      'rgt': 2,
      'tree_id': tree_id
    }))
    default_folder_id = res.inserted_primary_key[0]

    touched_users = set()

    if main_admin:
      session.execute(
        organizations_users_table.insert().values(
          user_id=main_admin.id,
          organization_id=organization_id,
          is_owner=True,
          order_num=0,
        )
      )
      touched_users.add(main_admin.id)

    for admin in other_admins:
      session.execute(
        organizations_users_table.insert().values(
          user_id=admin.id,
          organization_id=organization_id,
          is_owner=False,
          order_num=0,
        )
      )
      admin_permissions = admin.permissions or {}
      admin_permissions[f'can_edit_organization_cameras_{organization_id}'] = True
      admin_permissions[f'can_edit_organization_users_{organization_id}'] = True
      admin_permissions[f'can_view_organization_stats_{organization_id}'] = True
      session.execute(
        users_table.update(users_table.c.id==admin.id).values(
          permissions=admin_permissions
        )
      )
      touched_users.add(admin.id)

    if user.id not in touched_users:
      session.execute(
        organizations_users_table.insert().values(
          user_id=user.id,
          organization_id=organization_id,
          is_owner=False,
          order_num=0,
        )
      )

    user_permissions = user.permissions or {}
    user_permissions[f'can_edit_organization_cameras_{organization_id}'] = True
    user_permissions[f'can_edit_organization_users_{organization_id}'] = True
    user_permissions[f'can_view_organization_stats_{organization_id}'] = True
    session.execute(
      users_table.update(users_table.c.id==user.id).values(permissions=user_permissions)
    )

    session.execute(
      folder_users.insert().values(
        folder_id=default_folder_id, user_id=user.id
      )
    )

    for camera in owned_cameras.get_many_by('owner', user.login):
      # camera.organization = organization
      # camera.folder = default_folder
      session.execute(
        cloud_streams.update()
        .where(cloud_streams.c.name == camera.name)
        .values(folder_id=default_folder_id, organization_id=organization_id)
      )

  session.commit()


def downgrade():
  pass
